
/*************************************************
****      VEGA - BioDock output analyzer      ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/


#ifdef __WIN32__
#  include <windows.h>
#endif

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>

#ifdef __BORLANDC__
#  include <dir.h>
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "formats.h"

/**** Local prototypes ****/

static VG_BOOL BioDockGetFileName(char *, char *);

/**** Local definitions ****/

typedef struct __BdTraj {
  float             Dx, Dy, Dz;         /* Transaltion		       */
  float             Ax, Ay, Az;         /* Rotation		       */
} VG_BDTRAY;

typedef struct {
  VG_BDTRAY     *TrjPtr;                /* Trajectory pointer          */
  VG_BDTRAY     *CurFrm;                /* Current frame               */
  ATOMO         *LigBegAtm;             /* Ligand atom pointer         */
  VG_ULONG      LigTotAtm;              /* Ligand atoms                */
  XYZ           *LigCoord;              /* Ligand starting coordinates */
  ATOMO         *RecBegAtm;             /* Receptor atom pointer       */
  VG_ULONG      RecTotAtm;              /* Receptor atoms              */
  float         Rot[3];                 /* Rotation center             */
} VG_BDINFO;


/**** Free all resources ****/

void BioDockClose(TRJINFO *Info)
{
  if (Info -> FH) {
    if (((VG_BDINFO *)Info -> FH) -> TrjPtr)
      FREE(((VG_BDINFO *)Info -> FH) -> TrjPtr);

    if (((VG_BDINFO *)Info -> FH) -> LigCoord)
      FREE(((VG_BDINFO *)Info -> FH) -> LigCoord);

    FREE(Info -> FH);
  }
}


/**** Get the file name ****/

static VG_BOOL BioDockGetFileName(char *Dest, char *Src)
{
  char          *Ptr;

  if (Ptr = strchr(Src, ':')) {
    strcpy(Dest, Ptr + 3);
    for(Ptr = Dest + strlen(Dest); Ptr != Dest; --Ptr) {
      if (*Ptr == '\"') {
        *Ptr = 0;
        break;
      }
    } /* End of for */
  } else return CatErr(MSG_ERR_BIODOCK_STRUCTOUT);

  return TRUE;
}


/**** Open the trajectory file ****/

VG_BOOL BioDockOpen(char *FileName, TRJINFO *Info)
{
  ATOMO         *LastSeg;
  char          *Ptr, Buf[256], MolName[256];
  FILE          *FH;
  float         *Ener;
  VG_BDTRAY     *Trj;
  VG_ULONG      k, Line;
  XYZ           *LigPtr;

  VG_BOOL       Ret   = TRUE;

  /**** Change the current directory ****/

  for(Ptr = FileName + strlen(FileName) - 1; (*Ptr); --Ptr) {
    if ((*Ptr == '/') || (*Ptr == ':') || (*Ptr == '\\')) {
      k = *Ptr;
      *Ptr = 0;
      if (chdir(FileName) == -1) {
        PrintDosErr();
        return FALSE;
      }
      *Ptr = k;
      break;
    }
  } /* End of for */

  if ((Info -> FH = (FILE *)Alloca(sizeof(VG_BDINFO))) != NULL) {
    if ((FH = fopen(FileName, "r+")) != NULL)  {
      Line = 0;
      while((Ret) && (fgets(Buf, 255, FH))) {
        if (++Line == 1) {
          if (strncmp(Buf, "#BIODOCK_VER", 11)) {
            Ret = CatErr(MSG_ERR_BIODOCK_BIOOUT);
            break;
          }
        } else if (!strncmp(Buf, "#DOCKING_INFO", 13)) {
          if (SkipLines(FH, Buf, 2) &&
              BioDockGetFileName(MolName, Buf)) {
            ((VG_BDINFO *)Info -> FH) -> RecBegAtm = Loader(MolName,
                              &((VG_BDINFO *)Info -> FH) -> RecTotAtm, NULL, 1);
            if ((((VG_BDINFO *)Info -> FH) -> RecBegAtm) &&
                (((VG_BDINFO *)Info -> FH) -> RecBegAtm != (ATOMO *)-1) &&
                (((VG_BDINFO *)Info -> FH) -> RecBegAtm != (ATOMO *)-2)) {
              LastSeg = LastAtm;
              if (fgets(Buf, 255, FH)) {
                if (BioDockGetFileName(MolName, Buf)) {
                  ((VG_BDINFO *)Info -> FH) -> LigBegAtm = Loader(MolName,
                                  &((VG_BDINFO *)Info -> FH) -> LigTotAtm, NULL, 2);
                  if ((((VG_BDINFO *)Info -> FH) -> LigBegAtm) &&
                      (((VG_BDINFO *)Info -> FH) -> LigBegAtm != (ATOMO *)-1) &&
                      (((VG_BDINFO *)Info -> FH) -> LigBegAtm != (ATOMO *)-2)) {
                    TotalAtm = ((VG_BDINFO *)Info -> FH) -> RecTotAtm +
                               ((VG_BDINFO *)Info -> FH) -> LigTotAtm;
                    Info -> MolAtm = TotalAtm;
                    BegAtm = ((VG_BDINFO *)Info -> FH) -> RecBegAtm;
                    LastSeg -> Ptr = ((VG_BDINFO *)Info -> FH) -> LigBegAtm;

                    /**** Read the rotation center ****/

                    while((fgets(Buf, 255, FH)) && (*Buf != '*'));
                    if (!ferror(FH) || (!feof(FH))) {
                      if (Ret = SkipLines(FH, Buf, 3)) {
                        for(k = 0; (Ret) && (k < 3); ++k) {
                          if (fgets(Buf, 255, FH)) {
                            if (Ptr = strchr(Buf, ':'))
                              sscanf(Ptr + 1, "%f", &((VG_BDINFO *)Info -> FH) -> Rot[k]);
                            else Ret = CatErr(MSG_ERR_BIODOCK_STRUCTOUT);
                          } else Ret = PrintDosErr();
                        } /* End of for */
                      }

                      /**** Backup the ligand coordinates ****/

                      if (Ret) {
                        if ((((VG_BDINFO *)Info -> FH) -> LigCoord = (XYZ *)Alloca(((VG_BDINFO *)Info -> FH) -> LigTotAtm * sizeof(XYZ))) != NULL) {
                          LigPtr = ((VG_BDINFO *)Info -> FH) -> LigCoord;
                          for(LastSeg = ((VG_BDINFO *)Info -> FH) -> LigBegAtm;
                              LastSeg; LastSeg = LastSeg -> Ptr) {
                            LigPtr -> x = LastSeg -> x - ((VG_BDINFO *)Info -> FH) -> Rot[0];
                            LigPtr -> y = LastSeg -> y - ((VG_BDINFO *)Info -> FH) -> Rot[1];
                            LigPtr -> z = LastSeg -> z - ((VG_BDINFO *)Info -> FH) -> Rot[2];
                            ++LigPtr;
                          } /* End of for */
                        } else Ret = FALSE;
                      }
                    } else Ret = FALSE;
                  } else Ret = FALSE;
                } else Ret = FALSE;
              } else Ret = PrintDosErr();
            } else Ret = FALSE;
          } else Ret = FALSE;
        } else if (!strncmp(Buf, "#TRAJECTORY", 11)) {
          if (Ret = SkipLines(FH, Buf, 3)) {
            k = ftell(FH);

            /**** Count the frames ****/

            while((Ret) && (fgets(Buf, 255, FH)) && (*Buf != '#'))
              ++Info -> Frames;

            Info -> EneFrames = Info -> Frames;
            Info -> EneStep   = 0;
            Info -> StartTime = 0;
            Info -> FrmStep   = 0;
            Info -> Temp      = 0.0;
            if (((((VG_BDINFO *)Info -> FH) -> TrjPtr = (VG_BDTRAY *)Alloca(Info -> Frames * sizeof(VG_BDINFO))) != NULL) &&
                (Info -> Energy = (float *)Alloca(Info -> EneFrames * sizeof(float)))) {
              ((VG_BDINFO *)Info -> FH) -> CurFrm = ((VG_BDINFO *)Info -> FH) -> TrjPtr;
              fseek(FH, k, SEEK_SET);
              Ener = Info -> Energy;
              Trj  = ((VG_BDINFO *)Info -> FH) -> TrjPtr;
              while((Ret) && (fgets(Buf, 255, FH)) && (*Buf != '#')) {
                sscanf(Buf, "%*s %f %f %f %f %f %f %f",
                       Ener, &Trj -> Dx, &Trj -> Dy, &Trj -> Dz,
                       &Trj -> Ax , &Trj -> Ay , &Trj -> Az);
                ++Ener;
                ++Trj;
              } /* End of while */
            } else Ret = FALSE;
          }
        }
      } /* End of while */
      fclose(FH);
    } else Ret = PrintDosErr();
  } else Ret = FALSE;

  if (!Ret) BioDockClose(Info);

  return Ret;
}


/**** Read a frame ****/

VG_BOOL BioDockReadFrm(TRJINFO *Info)
{
  float         T;

  ATOMO         *Atm    = ((VG_BDINFO *)Info -> FH) -> LigBegAtm;
  VG_BDTRAY     *Trj    = ((VG_BDINFO *)Info -> FH) -> CurFrm;
  float		SinAx   = sin(Trj -> Ax);
  float		CosAx   = cos(Trj -> Ax);
  float		SinAy   = sin(Trj -> Ay);
  float		CosAy   = cos(Trj -> Ay);
  float		SinAz   = sin(Trj -> Az);
  float	        CosAz   = cos(Trj -> Az);
  XYZ           *LigPtr = ((VG_BDINFO *)Info -> FH) -> LigCoord;

  while(Atm) {
    Atm -> x = LigPtr -> x;
    Atm -> y = LigPtr -> y;
    Atm -> z = LigPtr -> z;

    T         = Atm -> x * CosAz - Atm -> y * SinAz;
    Atm -> y  = Atm -> x * SinAz + Atm -> y * CosAz;
    Atm -> x  = T;

    T         = Atm -> y * CosAx - Atm -> z * SinAx;
    Atm -> z  = Atm -> y * SinAx + Atm -> z * CosAx;
    Atm -> y  = T;

    T         = Atm -> x * CosAy - Atm -> z * SinAy;
    Atm -> z  = Atm -> x * SinAy + Atm -> z * CosAy;

    Atm -> x  = T + ((VG_BDINFO *)Info -> FH) -> Rot[0] + Trj -> Dx;
    Atm -> y +=     ((VG_BDINFO *)Info -> FH) -> Rot[1] + Trj -> Dy;
    Atm -> z +=     ((VG_BDINFO *)Info -> FH) -> Rot[2] + Trj -> Dz;

    Atm = Atm -> Ptr;
    ++LigPtr;
  }

  return TRUE;
}


/**** Seek a trajectory frame ****/

VG_BOOL BioDockSeekFrm(TRJINFO *Info, VG_LONG Frames, VG_LONG Mode)
{
  switch(Mode) {
  case SEEK_CUR:
    ((VG_BDINFO *)Info -> FH) -> CurFrm = ((VG_BDINFO *)Info -> FH) -> CurFrm + Frames;
    break;

  case SEEK_END:
    ((VG_BDINFO *)Info -> FH) -> CurFrm = ((VG_BDINFO *)Info -> FH) -> CurFrm +
                                          ((VG_BDINFO *)Info -> FH) -> LigTotAtm +
                                          Frames;
    break;

  case SEEK_SET:
    ((VG_BDINFO *)Info -> FH) -> CurFrm = ((VG_BDINFO *)Info -> FH) -> TrjPtr + Frames;
    break;
  } /* End of switch */

  return TRUE;
}

