
/*************************************************
****               VEGA - Charge              ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <math.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "formats.h"
#include "../LocaleLib/locale.h"


/**** Local prototypes ****/

static VG_BOOL ChargeTem(char *, ATOMO *, VG_ULONG);


/**** Dipole calculation ****/

float CalcDipole(register ATOMO *Atm, VG_ULONG Flags)
{
  float		Val;

  float		Xp = 0.0;
  float		Yp = 0.0;
  float		Zp = 0.0;
  float		Xn = 0.0;
  float		Yn = 0.0;
  float		Zn = 0.0;
  float		Cp = 0.0;
  float		Cn = 0.0;

  while(Atm) {

    /**** Skip waters (for trajectory analysis) ****/

   if (((Flags & SRF_FLG_NOWAT) &&
#ifdef LITTLE_ENDIAN
       (Atm -> ResName.L != 0x484F48)) ||
#else
       (Atm -> ResName.L != 0x00484F48)) ||
#endif
      ((Flags & SRF_FLG_SELONLY) && (!Atm -> Active))) {
      continue;
    }

    Val = (float)fabs(Atm -> Charge);
    if (Atm -> Charge > 0.0) {
      Xp += Atm -> x * Val;
      Yp += Atm -> y * Val;
      Zp += Atm -> z * Val;
      Cp += Val;
    } else {
      Xn += Atm -> x * Val;
      Yn += Atm -> y * Val;
      Zn += Atm -> z * Val;
      Cn += Val;
    }
    Atm = Atm -> Ptr;
  } /* End of while */

  if (Cp) {
    Xp /= Cp;
    Yp /= Cp;
    Zp /= Cp;
  }
  if (Cn) {
    Xn /= Cn;
    Yn /= Cn;
    Zn /= Cn;
  }

  return (SQR(Quad(Xp - Xn) + Quad(Yp - Yn) + Quad(Zp - Zn)) * DEBYE_CONST);
}


/**** Calculate a single point MEP ****/

float CalcSrfMep(register ATOMO *Atm, float X, float Y, float Z)
{
  register float         Dist;
  register float         Mep = 0;

  do {
    Dist = Quad(X - Atm -> x) + Quad(Y - Atm -> y) + Quad(Z - Atm -> z);
    Mep += Atm -> Charge / SQR(Dist);
    Atm  = Atm -> Ptr;
  } while(Atm);

  return Mep;
}


/**** Select charge routine ****/

VG_BOOL Charge(char *Template, VG_UWORD Type, ATOMO *InizAtm, VG_ULONG TotAtm)
{
  char			FFTemp[256];

  VG_BOOL               Ret = FALSE;

  PrintProg(MSG_CHARGE_ASSPROG);

  switch(Type) {
  case CHARGE_FRAG:
    Ret = ChargeTem(Template, InizAtm, TotAtm);
    break;
  case CHARGE_GAST:
    if ((Ret = GetPrgPath(FFTemp, TRUE))) {
      strcat(FFTemp, FF_UNIV_FILE);
      if ((Ret = AssignFF(FFTemp, InizAtm, TotAtm, TRUE))) {
        GetPrgPath(FFTemp, TRUE);
        strcat(FFTemp, "GASTEIGER.tem");
        Ret = Gasteiger(FFTemp, InizAtm, TotAtm);
      }
    }
    break;
  }

  return Ret;
}


/**** Calc the charge by composition ****/

VG_LONG ChargeBySeq(register ATOMO *Atm)
{
  VG_LONG       Chg         = 0;
  VG_LONG       PrecResName = 0;
  VG_LONG       PrecResSeq  = 0;
  VG_BYTE       PrecChainID = 0;

  while(Atm) {
    if ((PrecResName != Atm -> ResName.L) ||
        (PrecResSeq  != Atm -> ResSeq.L) ||
        (PrecChainID != Atm -> ChainID)) {
      if ((Atm -> ResName.L == *((VG_LONG *)"LYS")) ||
          (Atm -> ResName.L == *((VG_LONG *)"LYS+")) ||
          (Atm -> ResName.L == *((VG_LONG *)"ARG")) ||
          (Atm -> ResName.L == *((VG_LONG *)"ARG+"))) {
        ++Chg;
      } else if ((Atm -> ResName.L == *((VG_LONG *)"ASP")) ||
                 (Atm -> ResName.L == *((VG_LONG *)"ASP-")) ||
                 (Atm -> ResName.L == *((VG_LONG *)"GLU")) ||
                 (Atm -> ResName.L == *((VG_LONG *)"GLU-"))) {
        --Chg;
      }
      PrecResName = Atm -> ResName.L;
      PrecResSeq  = Atm -> ResSeq.L;
      PrecChainID = Atm -> ChainID;
    }
    Atm = Atm -> Ptr;
  } /* End of while */

  return Chg;
}


/**** Assign the charges and check it ****/

void ChargeMol(VG_UWORD Type)
{
  float         Fn;

  if (Charge(NULL, Type, BegAtm, TotalAtm)) {
    if (!ChkCharges(BegAtm, TotalAtm, NULL))
      CatErr(MSG_DLGCHG_ERR_NOTASS);
  } else CatErr(MSG_DLGCHG_ERR_NOTASS);
}


/**** Assign charges trough library of fragments ****/

static VG_BOOL ChargeTem(char *Template, ATOMO *InizAtm, VG_ULONG TotAtm)
{
  char			Buf[256], Com[16], ExtRes[12], Res[12];
  FILE			*FH;
  float			CharPar;
  VG_QCHAR		FastRes, Iiub;
  register ATOMO	*Atm;
  VG_ULONG		k;

  static const char	*Commands[] = { "RESNAME",
                                        "CHARGE",
                                        "DESC",
                                        "ENDDESC",
                                        "TITLE",
                                        ""
                                      };

  VG_BOOL			BegDesc = FALSE;
  VG_BOOL			Ret     = TRUE;
  VG_ULONG			Line    = 1;


  FindRes(InizAtm, TotAtm);

  if ((FH = fopen(Template, "r"))) {
    fgets(Buf, 255, FH);

    /**** Clean the charges ****/

    for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) Atm -> Charge = 10;

    while(fgets(Buf, 255, FH)) {
      if ((*Buf != '\n') && (*Buf != ';')) {
        if (*Buf == '#') {
          sscanf(Buf + 1, "%s", Com);
          for(k = 0;(*Commands[k]) && (strcasecmp(Com, Commands[k])); ++k);
          if (!*Commands[k]) {
            SetGlobErr(MSG_ERR_CHARGE_UNKCOM);
            ChargeErr(GetStr(MSG_ERR_CHARGE_UNKCOM), Buf, Line);
            break;
          }

          if (k == 2) BegDesc = TRUE;		/* Start the atom description  */
          else if (k == 3) BegDesc = FALSE;	/* Stop the atom description   */
            else if (!BegDesc) {
              switch(k) {

              case 0:	/* Residue name */
      	      	ExtRes[1] = 0;
      	      	ExtRes[2] = 0;
      	      	ExtRes[3] = 0;
              	FastRes.L = (VG_LONG)NULL;
              	sscanf(Buf, "%*s %s %s", Res, ExtRes);
              	strncpy(FastRes.C, ExtRes, 4);
              	break;

              case 1:	/* Total charge */
              	break;

              case 4:	/* Template name */
      	      	if (!GLOBSW_STDOUT) CatPrintf(stdout, MSG_CHARGE_DBNAME, Buf + 7);
              	break;
              }
            } else {
              SetGlobErr(MSG_ERR_CHARGE_ILLCOM);
              ChargeErr(GetStr(MSG_ERR_CHARGE_ILLCOM), Buf, Line);
              Ret = FALSE;
              break;
            }
        } else if (BegDesc) {
          Com[1] = 0;
          Com[2] = 0;
          Com[3] = 0;
          Iiub.L = (VG_LONG)NULL;
          sscanf(Buf, "%s", Com);
          strncpy(Iiub.C, Com, 4);
          sscanf(Buf + 61, "%f", &CharPar);
          for(Atm = InizAtm; Atm; Atm = Atm -> Ptr)
            if ((Atm -> ResName.L == FastRes.L) && (Atm -> Name.L == Iiub.L))
              Atm -> Charge = CharPar;
        }
      }
      ++Line;
    }
    fclose(FH);
  }

  /**** Check the charges ****/

  if (Ret) {
    BegDesc = TRUE;
    CharPar = 0;
    for(Atm = InizAtm; Atm; Atm = Atm -> Ptr)
      if (Atm -> Charge == 10) {
        Atm -> Charge = 0;
        SetGlobErr(MSG_ERR_CHARGE_UNKCHARGE);
        PrintAtmErr(GetStr(MSG_ERR_CHARGE_UNKCHARGE), Atm, FALSE);
        BegDesc = FALSE;
      } else CharPar += Atm -> Charge;
    if ((BegDesc) && (!GLOBSW_STDOUT))
      CatPrintf(stdout, MSG_CHARGE_TOTCHG, CharPar);
  }

  return Ret;
}


/**** Shows an error ****/

void ChargeErr(char *ErrMsg, char *Str, VG_ULONG Line)
{
  sprintf(ErrStr, "\n%s%d -> \"%s\"\n%s",
          GetStr(MSG_CHARGE_LINE), Line, ErrMsg, Str);

  if (ErrMode != VG_ERRMOD_QUIET)
    LocPrintf(stderr, "%s\n", ErrStr);
}


/**** Check if the charges are assigned ****/

VG_BOOL ChkCharges(register ATOMO *Atm, VG_ULONG TotAtm, float *TotCharge)
{
  VG_BOOL            Ret       = TRUE;
  register float     Chg       = 0;
  register VG_ULONG  Uncharged = 0;

  do {
    if (Atm -> Charge == 0.0) ++Uncharged;
    Chg += Atm -> Charge;
    Atm = Atm -> Ptr;
  } while(Atm);
  if (Uncharged != TotAtm) {
    if (TotCharge) *TotCharge = Chg;
  } else Ret = FALSE;

  return Ret;
}


/**** Find interesting residues ****/

VG_LONG FindRes(ATOMO *InizAtm, VG_ULONG TotAtm)
{
  char			*Ptr;
  VG_QCHAR              Res, ResName;
  register ATOMO	*Atm;
  register VG_UWORD	k;
  VG_WORD		Chk;

  ATOMO			*InizRes  = InizAtm;
  char			*CysCar   = "CSH";
  VG_LONG		TotCharge = 0;

  FFDESC        NTerm[]   = {{{"N"}, 3, 0, 0, 0},
                             {{"H"}, 1, 0, 0, 1},
                             {{"H"}, 1, 0, 0, 1},
                             {{"C"}, 4, 0, 0, 1},
                             {{"H"}, 1, 0, 0, 2},
                             {{"C"}, 3, 0, 0, 2},
                             {{"O"}, 1, 0, 0, 3},
                             {{"N"}, 3, 0, 0, 3}
                            };

  FFDESC        NTermC[]  = {{{"N"}, 4, 0, 0, 0},
                             {{"H"}, 1, 0, 0, 1},
                             {{"H"}, 1, 0, 0, 1},
                             {{"H"}, 1, 0, 0, 1},
                             {{"C"}, 4, 0, 0, 1},
                             {{"H"}, 1, 0, 0, 2},
                             {{"C"}, 3, 0, 0, 2},
                             {{"O"}, 1, 0, 0, 3},
                             {{"N"}, 3, 0, 0, 3}
                            };

  FFDESC        Arg[]     = {{{"C"}, 3, 0, 0, 0},
                             {{"N"}, 3, 0, 0, 1},
                             {{"N"}, 3, 0, 0, 1},
                             {{"N"}, 3, 0, 0, 1}
                            };

  FFDESC        AspGlu[]  = {{{"C"}, 3, 0, 0, 0},
                             {{"O"}, 1, 0, 0, 1},
                             {{"O"}, 1, 0, 0, 1},
                             {{"C"}, 4, 0, 0, 1},
                             {{"H"}, 1, 0, 0, 2},
                             {{"H"}, 1, 0, 0, 2}
                             };

  FFDESC        CTerm[]   = {{{"C"}, 3, 0, 0, 0},
                             {{"O"}, 1, 0, 0, 1},
                             {{"O"}, 2, 0, 0, 1},
                             {{"H"}, 1, 0, 0, 2},
                             {{"C"}, 4, 0, 0, 1},
                             {{"H"}, 1, 0, 0, 2},
                             {{"N"}, 3, 0, 0, 2},
                             {{"C"}, 3, 0, 0, 2}
                             };

  FFDESC        CTermC[]  = {{{"C"}, 3, 0, 0, 0},
                             {{"O"}, 1, 0, 0, 1},
                             {{"O"}, 1, 0, 0, 1},
                             {{"C"}, 4, 0, 0, 1},
                             {{"H"}, 1, 0, 0, 2},
                             {{"N"}, 3, 0, 0, 2},
                             {{"C"}, 3, 0, 0, 2}
                             };

  FFDESC        Lys[]     = {{{"N"}, 4, 0, 0, 0},
                             {{"H"}, 1, 0, 0, 1},
                             {{"H"}, 1, 0, 0, 1},
                             {{"H"}, 1, 0, 0, 1},
                             {{"C"}, 4, 0, 0, 1},
                             {{"H"}, 1, 0, 0, 2},
                             {{"H"}, 1, 0, 0, 2}
                            };

  if ((MatchMtx = (VG_UBYTE *)Alloca(TotAtm))) {

    PrintProg(MSG_CHARGE_PROGPMT);

    for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
      ResName.L = Atm -> ResName.L;
      if ((InizRes -> ResSeq.L != Atm -> ResSeq.L) ||
          (InizRes -> ResName.L != ResName.L)) InizRes = Atm;

      /**** Find charged GLU and ASP ****/

      if ((ResName.L == *(VG_LONG *)"GLU") || (ResName.L == *(VG_LONG *)"ASP")) {
        if (ChkFFAtm(Atm, &AspGlu[0])) {
          memset(MatchMtx, 0, TotAtm);
          if (MatchAtm(Atm, &AspGlu[1], &AspGlu[5])) {
            --TotCharge;
            ResName.C[3] = '-';
/*
            printf("%.2s %.2s %4.4s\n", Atm -> Conn[0] -> Elem.C,
              Atm -> Conn[1] -> Elem.C, InizRes -> ResSeq.C);
*/
            Atm = RenameRes(InizRes, ResName.L);
          }
        }
      } else if ((ResName.L == *(VG_LONG *)"LYS")) {

        /**** Find charged LYS ****/

      	if (ChkFFAtm(Atm, &Lys[0])) {
          memset(MatchMtx, 0, TotAtm);
          if (MatchAtm(Atm, &Lys[1], &Lys[6])) {
            ++TotCharge;
            Atm = RenameRes(InizRes, *(VG_LONG *)"LYS+");
          }
      	}
      } else if ((ResName.L == *(VG_LONG *)"ARG")) {

        /**** Find charged ARG ****/

      	if (ChkFFAtm(Atm, &Arg[0])) {
          memset(MatchMtx, 0, TotAtm);
          if (MatchAtm(Atm, &Arg[1], &Arg[3])) {
            ++TotCharge;
            Atm = RenameRes(InizRes, *(VG_LONG *)"ARG+");
          }
      	}
      } else

      /**** Find CYS and check sulfide bridges ****/

      	if ((ResName.L == *(VG_LONG *)"CYS")  &&  (Atm -> Elem.S == 0x5300)) {
      	  if (Atm -> NSost == 2) {
      	    Chk = 0;
      	    for(k = 0; k < 2; ++k)
      	      for(Ptr = CysCar; *Ptr; ++Ptr)
      	        if ((Atm -> Conn[k] -> Elem.C[0] == *Ptr) &&
      	            (!Atm -> Conn[k] -> Elem.C[1])) {
      	          ++Chk;
      	          break;
      	        }
      	    if (Chk != 2) --TotCharge;
      	    if ((Atm -> Conn[0] -> Elem.S == 0x4800) || (Atm -> Conn[1] -> Elem.S == 0x4800))
      	      Atm = RenameRes(InizRes, *(VG_LONG *)"CYH");

        } else if (Atm ->  NSost == 1) --TotCharge;
      }
    }

    /**** Find the chain terms ****/

    InizRes = InizAtm;
    for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
      ResName.L = Atm -> ResName.L;
      if ((InizRes -> ResSeq.L != Atm -> ResSeq.L) ||
          (InizRes -> ResName.L != ResName.L)) InizRes = Atm;
      for(k = 0; *AAResTab[k].C; ++k) {
        if (AAResTab[k].L == Atm -> ResName.L) {
          if (ChkFFAtm(Atm, &NTermC[0])) {
            memset(MatchMtx, 0, TotAtm);
            if (MatchAtm(Atm, &NTermC[1], &NTermC[8])) {
              Res.L = ResName.L;
              SetTypeRes(&Res, 'N');
              Atm = RenameRes(InizRes, Res.L);
              ++TotCharge;
            }
          } else if (ChkFFAtm(Atm, &NTerm[0])) {
            memset(MatchMtx, 0, TotAtm);
            if (MatchAtm(Atm, &NTerm[1], &NTerm[7])) {
              Res.L = ResName.L;
              SetTypeRes(&Res, 'n');
              Atm = RenameRes(InizRes, Res.L);
            }
          } else if (ChkFFAtm(Atm, &CTermC[0])) {
            memset(MatchMtx, 0, TotAtm);
            if (MatchAtm(Atm, &CTermC[1], &CTermC[6])) {
              Res.L = ResName.L;
              SetTypeRes(&Res, 'C');
              Atm = RenameRes(InizRes, Res.L);
      	      --TotCharge;
            } else if (MatchAtm(Atm, &CTerm[1], &CTerm[7])) {
              Res.L = ResName.L;
              SetTypeRes(&Res, 'c');
              Atm = RenameRes(InizRes, Res.L);
            }
          }
          break;
        }
      }
    }

    FREE(MatchMtx);
    RenameWat(InizAtm);
  }

  return TotCharge;
}


