
/*************************************************
****      VEGA - Connectivity Generator       ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/


#include <math.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "comerr.h"

#define  CONNTOL          1.2
#define  DISTCUTOFF       6.0

#ifdef LITTLE_ENDIAN
#  define  H_ELEMENT     0x48
#else
#  define  H_ELEMENT   0x4800
#endif


/**** Set covalent radii ****/

float *AssignCovRad(register ATOMO *Atomo, VG_ULONG TotAtomi)
{
  register ATMTAB    *Tab;
  register float     *CovRad;
  register VG_ULONG  k;

  if ((CovRad = (float *)Alloca(sizeof(float) * TotAtomi))) {
    for(k = 0;k < TotAtomi;++k) {
      for(Tab = AtmTable; *Tab -> Atom.C; ++Tab) {
        if (Tab -> Atom.S == Atomo -> Elem.S) {
          CovRad[k] = Tab -> CovRad;
          break;
        }
      } /* End of for */
      Atomo = Atomo -> Ptr;
    } /* End of for */
  }

  return CovRad;
}


/****           Set Van Der Waals radii         ****/
/**** If MaxRad is null, the maximum VDW radius ****/
/**** is not calculated.                        ****/

float *AssignVdwRad(register ATOMO *Atm, VG_ULONG TotAtm, float ProRad, float *MaxRad)
{
  VG_BOOL               WrtAtm;
  float                 *VdwRad;
  register float        *Rad;
  register VG_ULONG     k;

  VG_BOOL                  Err = FALSE;

  if ((VdwRad = (float *)Alloca(TotAtm * sizeof(float)))) {
    Rad     = VdwRad;
    if (MaxRad) *MaxRad = 0.0;
    while(Atm) {
      WrtAtm = FALSE;
      for(k = 0;*AtmTable[k].Atom.C;++k) {
      	if (AtmTable[k].Atom.S == Atm -> Elem.S) {
          *Rad  = AtmTable[k].VdwRad + ProRad;
          if ((MaxRad) && (*Rad > *MaxRad)) *MaxRad = *Rad;
          WrtAtm = TRUE;
          ++Rad;
          break;
      	}
      } /* End of element loop */
      if (!WrtAtm) {
        GlobErr = VG_PARET_MOLCONVDW;
        PrintAtmErr(GetStr(MSG_ERR_CONNECT_UNKVDW), Atm, FALSE);
        Err = TRUE;
      }
      Atm = Atm -> Ptr;
    } /* End of atom loop */
    if (Err) {
      FREE(VdwRad);
      VdwRad = NULL;
    }
  }

  return VdwRad;
}


/****           Set Van Der Waals radii         ****/
/**** If MaxRad is null, the maximum VDW radius ****/
/**** is not calculated.                        ****/

VG_BOOL AssignVdwNew(register ATOMO *Atm, float ProRad, float *MaxRad)
{
  VG_BOOL               WrtAtm;
  register ATMTAB       *Tab;

  VG_BOOL               Ret = TRUE;

  if (MaxRad) *MaxRad = 0.0;
  while(Atm) {
    WrtAtm = FALSE;
    for(Tab = AtmTable;Tab -> Atom.S;++Tab) {
      	if (Tab -> Atom.S == Atm -> Elem.S) {
          Atm -> Rad  = Tab -> VdwRad + ProRad;
          if ((MaxRad) && (Atm -> Rad > *MaxRad)) *MaxRad = Atm -> Rad;
          WrtAtm = TRUE;
          break;
      	}
    } /* End of element loop */
    if (!WrtAtm) {
      GlobErr = VG_PARET_MOLCONVDW;
      Ret = PrintAtmErr(GetStr(MSG_ERR_CONNECT_UNKVDW), Atm, FALSE);
    }
    Atm = Atm -> Ptr;
  } /* End of atom loop */

  return Ret;
}


/**** Calculate the connectivity ****/

VG_BOOL Connect(register ATOMO *Atomo1, VG_ULONG TotAtomi)
{
  float                 *CovRadTbl;
  register ATOMO        *Atomo2;
  register float        *CovRad1, *CovRad2;
  register float        Dist, Tmp;

  VG_BOOL               Ret = TRUE;

  PrintProg(MSG_CONNECT_PROGPMT);

  if ((CovRadTbl = (float *)AssignCovRad(Atomo1, TotAtomi))) {

    /**** Check the connectivity ****/

    CovRad1 = CovRadTbl;
    while((Ret) && Atomo1 -> Ptr) {
      if (!((Atomo1 -> Elem.S == H_ELEMENT) && (Atomo1 -> NSost))) {
        CovRad2 = CovRad1 + 1;
      	for(Atomo2  = Atomo1 -> Ptr;(Atomo2) && (Ret); Atomo2 = Atomo2 -> Ptr) {
          Dist = Quad(Atomo1 -> x - Atomo2 -> x) + Quad(Atomo1 -> y - Atomo2 -> y) +
                 Quad(Atomo1 -> z - Atomo2 -> z);
          if (Dist <= DISTCUTOFF) {
            Tmp = (*CovRad1 + *CovRad2) * CONNTOL;
            if (Dist <= Quad(Tmp)) {
              if ((Atomo1 -> NSost < MAXBOND) && (Atomo1 -> NSost < MAXBOND)) {
                Atomo1 -> Conn[(int)Atomo1 -> NSost] = Atomo2;
                Atomo2 -> Conn[(int)Atomo2 -> NSost] = Atomo1;
                ++Atomo1 -> NSost;
                ++Atomo2 -> NSost;
              } else Ret = CatErr(MSG_ERR_CONNECT_UNACCBD);
            }
          }
          ++CovRad2;
      	} /* End of Atomo2 for */
      }
      Atomo1 = Atomo1 -> Ptr;
      ++CovRad1;
    } /* End of Atomo1 while */
    FREE(CovRadTbl);
  } else {
    PDBFree(Atomo1);
    Ret = FALSE;
  }

  return Ret;
}

/**** Normalize the coordinates ****/

void Normalize(register ATOMO *Atomo, VG_ULONG TotAtomi)
{
  register VG_SURFACE   *Dot;
  XYZ                   GeoCent;

  PrintProg(MSG_CONNECT_NORMPROG);

  CalcGeoCent(Atomo, &GeoCent, FALSE);

  /**** Translate all atoms ****/

  while(Atomo) {
    Atomo -> x -= GeoCent.x;
    Atomo -> y -= GeoCent.y;
    Atomo -> z -= GeoCent.z;
    Atomo       = Atomo -> Ptr;
  }

  /**** Translate the surface ****/

  if (BegSrf) {
    for(Dot = BegSrf; Dot; Dot = Dot -> Ptr) {
      Dot -> x -= GeoCent.x;
      Dot -> y -= GeoCent.y;
      Dot -> z -= GeoCent.z;
    } /* End of for */
  }
}


/**** Get the atomic number ****/

VG_UWORD GetAtmNum(register VG_UWORD Elem)
{
  register ATMTAB       *Tab = AtmTable;
  register VG_UWORD     Ret  = 0;

  while(Tab -> Atom.S) {
    if (Tab -> Atom.S == Elem) {
      Ret = Tab -> AtmNum;
      break;
    }
    ++Tab;
  } /* End of while */

  return Ret;
}
