
/*************************************************
****  VEGA - Quanta DCD Trajectory analyzer   ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/

/*
 * VEGA writes the binary plot in big endian format.
 * Now is compatible with big and little endian DCD files
 */

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "formats.h"

#ifdef WIN32
#include <conio.h>
#endif


/**** Header for DCD trajectory ****/

typedef struct {
  VG_QCHAR          Hdr;            /* Header (CORD)                     */
  VG_ULONG          TotFrames;      /* Number of frames                  */
  VG_ULONG          StartTime;      /* Starting time of first frame (fs) */
  VG_ULONG          FrameStep;      /* Frame step                        */
  VG_ULONG          Res1[5];        /* Not used                          */
  VG_ULONG          NFixed;         /* Number of fixed atoms             */
  VG_ULONG          Res2;           /* Not used                          */
  VG_ULONG          Crystal;        /* Crystal flag                      */
  VG_ULONG          Res3[8];        /* Not used                          */
  VG_ULONG          Version;        /* CHARMm version                    */
} DCDHDR;

/**** Local variables ****/

static VG_BOOL                  SwapEndian;

/**** Local proptotypes ****/

static VG_BOOL DCDRead(FILE *, VG_ULONG, TRJPAR *, ATOMO *);


/**** Close the trajectory ****/

void DCDClose(TRJINFO *Info)
{
  if (Info -> FH) fclose(Info -> FH);
}


/**** Load the associated .ene file ****/

VG_BOOL DCDLoadEne(FILE *FH, TRJINFO *Info)
{
  char          Buf[256];
  float         *Ptr;
  VG_ULONG      k;

  VG_BOOL       Ret = TRUE;
  VG_ULONG      Frm = 0;

  /**** Count the frames ****/

  while((fgets(Buf, 255, FH)) &&
        (fgets(Buf, 255, FH)) &&
        (fgets(Buf, 255, FH))) {
    ++Info -> EneFrames;
  } /* End of while */

  if ((Info -> EneFrames) &&
      (Info -> Energy = (float *)Alloca(Info -> EneFrames * sizeof(float)))) {
    Frm = 0;
    Ptr = Info -> Energy;
    fseek(FH, 0, SEEK_SET);
    while(fgets(Buf, 256, FH)) {
      ++Frm;
      sscanf(Buf, "%d %*s %*s %*s %f", &k, Ptr++);

      /**** Start frame and step info ****/

/*
      if (Frm == 1) Info -> StartTime = k;
*/
      if (Frm == 2) Info -> EneStep = k - Info -> StartTime;

      if ((!fgets(Buf, 255, FH)) ||
          (!fgets(Buf, 255, FH))) break;
    } /* End of while */
  } else Ret = FALSE;

  return Ret;
}



/**** Load the associated NAMD .out file ****/

VG_BOOL DCDLoadNamdOut(FILE *FH, TRJINFO *Info)
{
  char          Buf[256];
  float         *Ptr;
  float         EneKin, EneTot;
  VG_ULONG      k;

  VG_BOOL       FrmStep = FALSE;
  VG_BOOL       EneStep = FALSE;
  VG_BOOL       Ret     = TRUE;
  VG_BOOL       Temp    = FALSE;
  VG_ULONG      Frm     = 0;

  /**** Count the frames ****/

  while(fgets(Buf, 255, FH)) {
    if (!strncmp(Buf, "ENERGY:", 7)) {
      ++Info -> EneFrames;

      /**** Trajectory refresh ****/

    } else if ((!FrmStep) && (!strncmp(Buf, "Info: DCD FREQUENCY", 19))) {
      sscanf(Buf + 29, "%d", &Info -> FrmStep);
      FrmStep = TRUE;

      /**** Log refresh ****/

    } else if ((!EneStep) && (!strncmp(Buf, "Info: ENERGY OUTPUT STEPS", 25))) {
      sscanf(Buf + 29, "%d", &Info -> EneStep);
      EneStep = TRUE;

      /**** Temperature ****/

    } else if ((!Temp) && (!strncmp(Buf, "Info: INITIAL TEMPERATURE", 25))) {
      sscanf(Buf + 29, "%f", &Info -> Temp);
      Temp = TRUE;
    }
  } /* End of while */

  if (Info -> EneFrames) {
    --Info -> EneFrames;
    if ((Info -> EneFrames) &&
        (Info -> Energy = (float *)Alloca(Info -> EneFrames * sizeof(float)))) {
      Frm = 0;
      Ptr = Info -> Energy;
      fseek(FH, 0, SEEK_SET);
      while(fgets(Buf, 256, FH)) {
        if (!strncmp(Buf, "ENERGY:", 7)) {
          sscanf(Buf, "%*s %d %*s %*s %*s %*s %*s %*s %*s %*s %f %f",
                 &k, &EneKin, &EneTot);
          if (k) {
            ++Frm;
            if (Frm == 1) Info -> StartTime = k;
            else if (Frm == 2) Info -> EneStep = k - Info -> StartTime;
            *Ptr++ = EneTot - EneKin;
          }
        }
      } /* End of while */
    } else Ret = FALSE;
  } else Ret = FALSE;

  return Ret;
}


/**** Open the trajectory file ****/

VG_BOOL DCDOpen(char *FileName, TRJINFO *Info)
{
  char          Riga[32];
  DCDHDR        Hdr;
  FILE          *FH2;
  VG_BOOL       Ret;
  VG_UWORD      LogFormat;

  if ((Info -> FH = fopen(FileName, "rb"))) {

    /**** Check the endian ****/

    if (fread(Riga, sizeof(int), 1, Info -> FH) == 1) {
      if (Riga[3] == 84) SwapEndian = FALSE; /* Big Endian    */
      else SwapEndian = TRUE  ;              /* Little Endian */

#ifdef LITTLE_ENDIAN
      SwapEndian = !SwapEndian;
#endif

      if (fread(&Hdr, sizeof(DCDHDR), 1, Info -> FH) == 1) {
        fseek(Info -> FH, sizeof(int), SEEK_CUR);

        if (SwapEndian) {
          Swap(&Hdr.TotFrames);
          Swap(&Hdr.StartTime);
          Swap(&Hdr.FrameStep);
/*
          Swap(&Hdr.NFixed);
*/
        }
        Info -> Frames    = Hdr.TotFrames;
        Info -> StartTime = Hdr.StartTime;
        Info -> FrmStep   = Hdr.FrameStep;

        if ((Ret = ForSeek(Info -> FH, 1, SwapEndian))) {
          if ((Ret = ForRead(Info -> FH, &Info -> MolAtm, sizeof(VG_ULONG)))) {
            if (SwapEndian)
              Swap(&Info -> MolAtm);

            /**** Calculates the real number of frames ****/

            Info -> StartData = ftell(Info -> FH);
            fseek(Info -> FH, 0, SEEK_END);
            Info -> Frames = (ftell(Info -> FH) - Info -> StartData) /
                             (sizeof(float) * 3 * Info -> MolAtm +
                              sizeof(VG_ULONG) * 6);
            fseek(Info -> FH, Info -> StartData, SEEK_SET);

            /**** Read .ENE file (if exists) ****/

            ChangeExt(FileName, "ENE", Riga);
            FH2 = OpenPlot(FileName, &LogFormat);
            if (FH2 == NULL) {
              ChangeExt(FileName, "out", NULL);
              FH2 = OpenPlot(FileName, &LogFormat);
            }
            ChangeExt(FileName, Riga, NULL);
            if (FH2 != NULL) {
              switch(LogFormat) {
              case FPLOT_CHARMMENE:
                DCDLoadEne(FH2, Info);
                break;

              case FPLOT_NAMDOUT:
                DCDLoadNamdOut(FH2, Info);
                break;
              } /* End of switch */
              PkClose(FH2);
            } /* End of switch */
          }
        }
      } else Ret = PrintDosErr();
    } else Ret = PrintDosErr();
  } else Ret = PrintDosErr();

  if ((!Ret) && (Info -> FH)) fclose(Info -> FH);

  return Ret;
}

/**** Read specified atoms of a trajectory frame ****/

static VG_BOOL DCDRead(FILE *FH, VG_ULONG TotAtm, TRJPAR *Par, ATOMO *Frm)
{
  VG_ULONG         AttPos;
  VG_UWORD         k;

  VG_BOOL          Ret     = TRUE;
  VG_ULONG         PrecPos = 0;

  /**** Read X coordinates ****/

  fseek(FH, sizeof(VG_ULONG), SEEK_CUR);
  for(k = 0; (Ret) && (k < Par -> Num); ++k) {
    AttPos = (Par -> Atm[k].Num - 1) * sizeof(float);
    fseek(FH, AttPos - PrecPos, SEEK_CUR);
    if (fread(&Frm[k].x, sizeof(float), 1, FH) == 1) {
      if (SwapEndian) Swap(&Frm[k].x);
      PrecPos = AttPos + sizeof(float);
    } else Ret = PrintDosErr();
  }

  /**** Read Y coordinates ****/

  if (Ret) {
    fseek(FH, (TotAtm - Par -> Atm[k - 1].Num) * sizeof(float) +
              sizeof(VG_ULONG) * 2, SEEK_CUR);
    PrecPos = 0;
    for(k = 0; (Ret) && (k < Par -> Num); ++k) {
      AttPos = (Par -> Atm[k].Num - 1) * sizeof(float);
      fseek(FH, AttPos - PrecPos, SEEK_CUR);
      if (fread(&Frm[k].y, sizeof(float), 1, FH) == 1) {
        if (SwapEndian) Swap(&Frm[k].y);
        PrecPos = AttPos + sizeof(float);
      } else Ret = PrintDosErr();
    }

    /**** Read Z coordinates ****/

    if (Ret) {
      fseek(FH, (TotAtm - Par -> Atm[k - 1].Num) * sizeof(float) +
            sizeof(VG_ULONG) * 2, SEEK_CUR);
      PrecPos = 0;
      for(k = 0; (Ret) && (k < Par -> Num); ++k) {
        AttPos = (Par -> Atm[k].Num - 1) * sizeof(float);
        fseek(FH, AttPos - PrecPos, SEEK_CUR);
        if (fread(&Frm[k].z, sizeof(float), 1, FH) == 1) {
          if (SwapEndian) Swap(&Frm[k].z);
          PrecPos = AttPos + sizeof(float);
        } else Ret = PrintDosErr();
      }
      fseek(FH, (TotAtm - Par -> Atm[k - 1].Num) * sizeof(float) +
            sizeof(VG_ULONG), SEEK_CUR);
    }
  }

  return Ret;
}


/**** Read all atoms of a trajectory frame ****/

VG_BOOL DCDReadFrm(FILE *FH, ATOMO *InizAtm)
{
  register ATOMO        *Atm;

  VG_BOOL               Ret = TRUE;

  /**** Read X coordinates ****/

  fseek(FH, sizeof(VG_ULONG), SEEK_CUR);
  for(Atm = InizAtm;(Ret) && (Atm); Atm = Atm -> Ptr) {
    if (fread(&Atm -> x, sizeof(float), 1, FH) != 1)
      Ret = PrintDosErr();
    else if (SwapEndian) Swap(&Atm -> x);
  } /* End of atom loop */

  if (Ret) {

    /**** Read Y coordinates ****/

    fseek(FH, sizeof(VG_ULONG) * 2, SEEK_CUR);
    for(Atm = InizAtm;(Ret) && (Atm); Atm = Atm -> Ptr) {
      if (fread(&Atm -> y, sizeof(float), 1, FH) != 1)
        Ret = PrintDosErr();
      else if (SwapEndian) Swap(&Atm -> y);
    }
    if (Ret) {

      /**** Read Z coordinates ****/

      fseek(FH, sizeof(VG_ULONG) * 2, SEEK_CUR);
      for(Atm = InizAtm;(Ret) && (Atm); Atm = Atm -> Ptr) {
        if (fread(&Atm -> z, sizeof(float), 1, FH) != 1)
          Ret = PrintDosErr();
        else if (SwapEndian) Swap(&Atm -> z);
      }
      if (Ret) fseek(FH, sizeof(VG_ULONG), SEEK_CUR);
    }
  }

  return Ret;
}


/**** Seek a trajectory frame ****/

VG_BOOL DCDSeekFrm(TRJINFO *Info, VG_LONG Frames, VG_LONG Mode)
{
  VG_LONG      Len;

  if (Mode == SEEK_SET) Len = Info -> StartData;
  else Len = 0;

  Len += (sizeof(VG_ULONG) * 2 + sizeof(float) * Info -> MolAtm) * 3 * Frames;
  fseek(Info -> FH, Len, Mode);

  return TRUE;
}
