
/*************************************************
****      VEGA - Force Field Attribution      ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/


/*
 * This code assigns a selected force field template in ATDL format to the
 * loaded molecule.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"

#define  FFDESC_LEN		sizeof(FFDESC)
#define  MAXFFBOND		4
#define  MAXFFSCORE		5
#define  MAXRING		6
#define  MAXSUBLEVELS           256
#define  ORDMTX_LEN		(sizeof(OrdMtxL) / sizeof(LONG))
#define  TOTHUCK		(sizeof(PiMtr) / sizeof(HUCK))


/**** Local prototypes ****/

static VG_BOOL FFError(FFPAR *, VG_ULONG);


/**** Check the file template ****/

VG_BOOL ChkTempFF(char *TempName, char *FField)
{
  char                  Buf[12];
  FILE                  *TEST;
  VG_ULONG              Len;

  VG_BOOL               Ret = TRUE;

  /**** Set the file name ****/

  GetPrgPath(FField, TRUE);
  Len = strlen(FField);
  strcat(FField, TempName);
  Upper(FField + Len - 1);
  strcat(FField, ".tem");

  if ((TEST = fopen(FField, "r"))) {
    fgets(Buf, 12, TEST);
    if (strncasecmp(Buf, "#TemplateFF", 11))
      Ret = CatErr(MSG_ERR_MAIN_FFNOTSTD);
    fclose(TEST);
  } else Ret = CatErr(MSG_ERR_MAIN_FFNOTFOUND);

  return Ret;
}


/**** Assign the force field ****/

VG_BOOL AssignFF(char *FFFile, ATOMO *InizAtm, VG_ULONG TotAtomi, VG_BOOL Quiet)
{
  char			Buf[256], Tmp[4];
  FFDESC		*FFArgs;
  FFPAR			ParFF;
  VG_QCHAR		FFType;
  register ATOMO        *Atm;
  VG_UBYTE		SubCount;
  VG_ULONG		k;
  VG_UWORD		j;

  FFDESC		*InizFFArgs = NULL;
  VG_BOOL		Ret         = TRUE;


  if ((MatchMtx = (VG_UBYTE *)Alloca(TotAtomi))) {
    if ((InizFFArgs = (FFDESC *)Alloca(MAXSUBLEVELS * FFDESC_LEN))) {
      if (!Quiet) PrintProg(MSG_FFIELD_FIXPROG);

      /**** Clean the force field ****/

      Atm  = InizAtm;
      while(Atm) {
        Atm -> Pot.L  = (VG_ULONG)NULL;
        *Atm -> Pot.C = '?';
        Atm = Atm -> Ptr;
      }

      ParFF.Line = 2;
      if ((ParFF.FFIN = fopen(FFFile, "r"))) {
        fgets(Buf, 256, ParFF.FFIN);
        while((Ret) && (fgets(Buf, 255, ParFF.FFIN))) {

          /**** Filter comments and blank lines ****/

          *Tmp = 0;
          sscanf(Buf, "%3s", Tmp);
          if ((*Tmp) && (*Tmp != ';')) {

            /**** Count the number of arguments ****/

            j             = 0;
            k             = 0;
            ParFF.TotArgs = (VG_ULONG)NULL;
            ParFF.Beg     = NULL;
            FFArgs        = InizFFArgs;
            for(ParFF.CPtr = Buf;(*ParFF.CPtr && (*ParFF.CPtr != ';'));++ParFF.CPtr) {
              if (!ParFF.Beg) {
                if ((*ParFF.CPtr != ' ') && (*ParFF.CPtr != '\t') && (*ParFF.CPtr != '(') && (*ParFF.CPtr != ')')) ParFF.Beg = ParFF.CPtr;
              } else switch (*ParFF.CPtr) {
              case '(':
              case ')':
              case ' ':
              case '\n':
              case '\t':
                ++ParFF.TotArgs;
                ParFF.Beg = NULL;
              };
              switch (*ParFF.CPtr) {
              case '(':
                ++j;
                ++k;
                break;
              case ')':
                --k;
                break;
              }
            }
            if (ParFF.TotArgs < 2) {
              ParFF.Beg = NULL;
              Ret = FFError(&ParFF, MSG_ERR_FFIELD_ILLNUM);
              break;
            }
            if (k) {
              ParFF.Beg = NULL;
              Ret = FFError(&ParFF, MSG_ERR_FFIELD_UNBPAR);
              break;
            }
            if (j > MAXSUBLEVELS) {
              ParFF.Beg = NULL;
              Ret = FFError(&ParFF, MSG_ERR_FFIELD_SUBLVL);
              break;
            }

            /**** Read the arguments ****/

            --ParFF.TotArgs;
            ParFF.Args = (VG_ULONG)NULL;
            ParFF.Beg  = NULL;
            SubCount   = 0;

            for(ParFF.CPtr = Buf;((Ret) && (*ParFF.CPtr) && (*ParFF.CPtr != ';'));++ParFF.CPtr) {
              if (*ParFF.CPtr == '(') ++SubCount;
              if (!ParFF.Beg) {
	        if ((*ParFF.CPtr != ' ') && (*ParFF.CPtr != '\t') &&
	            (*ParFF.CPtr != '(') && (*ParFF.CPtr != ')')) ParFF.Beg = ParFF.CPtr;
              } else switch (*ParFF.CPtr) {
              case '(':
              case ')':
              case ' ':
              case '\n':
              case '\t':
                if (ParFF.Args) FFArgs -> NSub = SubCount;
                if (*ParFF.CPtr == ')') --SubCount;
                *ParFF.CPtr = 0;

                switch (ParFF.Args) {

                case 0:	/* Force field atom name */

                  if ((ParFF.CPtr - ParFF.Beg) > 5)
                    Ret = FFError(&ParFF, MSG_ERR_FFIELD_ATYPTOOLONG);
                  else {
/*
                    memcpy(&FFType.L, ParFF.Beg, 4);
*/
                    FFType.L = 0;
                    for(k = 0; (k < 4) && (ParFF.Beg[k] != ' ') &&
                      (ParFF.Beg[k] != '\t') && (ParFF.Beg[k] != '\n'); ++k)
                      FFType.C[k] = ParFF.Beg[k];
                  }
                  break;

                case 1:	/* Atom description */

                  if ((ParFF.CPtr - ParFF.Beg) > 6)
                    Ret = FFError(&ParFF, MSG_ERR_FFIELD_INCATM);
                  else {
                    Ret = ToDesc(&ParFF, InizFFArgs, ParFF.Beg);
                    FFArgs = InizFFArgs + 1;
                  }
                  break;

                default: /* Connected atom description */

                  if ((ParFF.CPtr - ParFF.Beg) > 7)
                    Ret = FFError(&ParFF, MSG_ERR_FFIELD_INCSUBATM);
                  else {
                    Ret = ToDesc(&ParFF, FFArgs, ParFF.Beg);
                    ++FFArgs;
                  }
                }
                ++ParFF.Args;
                ParFF.Beg = NULL;
              };
            }

            if (Ret) {
              Atm  = InizAtm;
              while(Atm) {
                if ((*Atm -> Pot.C == '?') && (ChkFFAtm(Atm, InizFFArgs))) {
                  if (ParFF.TotArgs > 1) {
                    memset(MatchMtx, 0, TotAtomi);
                    if (MatchAtm(Atm, InizFFArgs + 1, InizFFArgs + ParFF.TotArgs - 1)){
/*
                      printf("%4.4s -> %d %4.4s %2.2s %3.3s_%-4.4s\n",
                             FFType.C, Atm -> Num, Atm -> Name.C, Atm -> Elem.C,
                        	   Atm -> ResName.C, Atm -> ResSeq.C);
*/
                      Atm -> Pot.L = FFType.L;
                    }
                  } else Atm -> Pot.L = FFType.L;
                }
                Atm = Atm -> Ptr;
              }
            }
          }
          ++ParFF.Line;
        }
        fclose(ParFF.FFIN);
      } else Ret = CatErr(MSG_ERR_FFIELD_NOTFOUND);
      FREE(InizFFArgs);
    }
    FREE(MatchMtx);

    /**** Check if the force field is correctly assigned ****/

    if (Ret) {
      Atm  = InizAtm;
      while(Atm) {
      	if (*Atm -> Pot.C == '?') {
          SetGlobErr(MSG_ERR_FFIELD_CANTASSIGN);
      	  Ret = PrintAtmErr(GetStr(MSG_ERR_FFIELD_CANTASSIGN), Atm, FALSE);
        }
      	Atm = Atm -> Ptr;
      }
    }
  } else Ret = FALSE;

  return Ret;
}


/**** Convert an atom into atdl string ****/

void AtmToAtdl(char *Str, ATOMO *Atm)
{
  Str[0] = Atm -> Elem.C[0];
  Str[1] = Atm -> Elem.C[1];
  if (!Str[1]) Str[1] = '-';
  Str[2] = (char)(Atm -> NSost + '0');
  if (Atm -> Ring < 0)
    Str[4] = '1';
  else
    Str[4] = '0';

  Str[3] = (char)(abs(Atm -> Ring) + '0');
  Str[5] = 0;
}


/**** Match the atom with force field template ****/

VG_UWORD MatchAtm(register ATOMO *Atm, FFDESC *FFStart, FFDESC *FFEnd)
{
  ATOMO             *TmpAtm;
  register FFDESC   *FFArg;
  VG_UWORD          k;

  register FFDESC   *PrecFFArg = NULL;
  VG_BOOL           Match      = FALSE;
  VG_BOOL           Flag       = FALSE;
  VG_ULONG          TmpNum     = 0;
  VG_UWORD          Ret        = 0;

  for(FFArg = FFStart;FFArg <= FFEnd; ++FFArg) {
    for(k = 0;k < Atm -> NSost; ++k) {
      TmpAtm = Atm -> Conn[k];
      TmpNum = TmpAtm -> Num - 1;
      if ((!MatchMtx[TmpNum]) && (ChkFFAtm(TmpAtm, FFArg))) {
        PrecFFArg = FFArg;
        if ((FFArg != FFEnd) && (FFArg -> NSub < (FFArg + 1) -> NSub)) {
          Match  = TRUE;
          Ret    = MatchAtm(TmpAtm, FFArg + 1, FFEnd);
          if (!Ret) return 0;
          FFArg += Ret;
          Flag   = TRUE;
        } else {
          ++Ret;
          Match = FALSE;
          Flag  = TRUE;
        }
        break;
      }
    }

    if (Flag) {
      MatchMtx[TmpNum] = TRUE;
/*
      printf("%d %.4s %2.2s %d\n",
             TmpAtm -> Num, TmpAtm -> ResName.C, TmpAtm -> Name.C, PrecFFArg -> NSub);
*/
      if (FFArg != FFEnd) {
        if (Match) {
          if (((FFArg + 1) <= FFEnd) && (PrecFFArg -> NSub == (FFArg + 1) -> NSub))
            Flag = FALSE;
        } else {
          if (PrecFFArg -> NSub == (PrecFFArg + 1) -> NSub) Flag = FALSE;
          if (PrecFFArg -> NSub > (PrecFFArg + 1) -> NSub) break;
        }
/*
        if ((Match) && ((FFArg + 2) <= FFEnd) &&
          ((FFArg + 1) -> NSub == (FFArg + 2) -> NSub)) Flag = TRUE;
        else {
          if (PrecFFArg -> NSub == (PrecFFArg + 1) -> NSub) Flag = FALSE;
          if (PrecFFArg -> NSub > (PrecFFArg + 1) -> NSub) break;
        }
*/
      }
    } else {
      Ret = 0;
      break;
    }
  }

  return Ret;
}


/**** Check the atom matching ****/

VG_BOOL ChkFFAtm(register ATOMO *Atm, register FFDESC *FFArg)
{
  VG_BOOL		Ret = FALSE;


#ifndef LITTLE_ENDIAN
  if ((Atm -> Elem.S == FFArg -> Atom.S) || (*FFArg -> Atom.C == 'X') ||
      ((*FFArg -> Atom.C == '#') && (Atm -> Elem.S != 0x4800)) ||
      ((*FFArg -> Atom.C == '$') && (Atm -> Elem.S != 0x4800) && (Atm -> Elem.S != 0x4300))) {
#else
  if ((Atm -> Elem.S == FFArg -> Atom.S) || (*FFArg -> Atom.C == 'X') ||
      ((*FFArg -> Atom.C == '#') && (Atm -> Elem.S != 0x0048)) ||
      ((*FFArg -> Atom.C == '$') && (Atm -> Elem.S != 0x0048) && (Atm -> Elem.S != 0x0043))) {
#endif

    if ((Atm -> NSost == FFArg -> NSost) || (FFArg -> NSost == 9)) {
      if (FFArg -> Ring) {
        if ((FFArg -> Ring == 9) && (Atm -> Ring)) Ret = TRUE;
        else if (abs(Atm -> Ring) == FFArg -> Ring) Ret = TRUE;
        if ((Ret) && (FFArg -> Arom) && (Atm -> Ring > 0)) Ret = FALSE;
      } else Ret = TRUE;
    }
  }

  return Ret;
}


/**** Show an error in force field file ****/

static VG_BOOL FFError(FFPAR *ParFF, VG_ULONG StrCod)
{
  if (ParFF -> Beg)
    CatErr(MSG_ERR_FFIELD_FFERRLINE, ParFF -> Line, ParFF -> Beg, GetStr(StrCod));
  else
    CatErr(MSG_ERR_FFIELD_FFERRLINETEMP, ParFF -> Line, GetStr(StrCod));
  SetGlobErr(StrCod);

  return FALSE;
}


/**** Find the rings ****/

void FindRing(ATOMO *InizAtm)
{
  ATOMO             *AtmMtr[MAXRING];
  VG_LONG           k, N3, Ring, TotPi;
  VG_LONG           NSost0, NSost1, NSost2;
  VG_LONG           NSost3, NSost4, NSost5;
  register ATOMO    *Atm, *Temp;
  VG_WORD           j;

  static HUCK PiMtr[] = {{{"C"}, 3, 1},	/* Don't change the order */
                         {{"N"}, 3, 2},
                         {{"N"}, 2, 1},
                         {{"O"}, 2, 1},
                         {{"S"}, 2, 1}
                        };

  PrintProg(MSG_FFIELD_RINGPROG);

  Atm = InizAtm;
  while(Atm) {
    if (Atm -> NSost > 1) {
      Ring    = 0;
      *AtmMtr = Atm;
      for(NSost0 = 0;NSost0 < AtmMtr[0] -> NSost;++NSost0) {
        Temp = AtmMtr[0] -> Conn[NSost0];
        if (Temp -> NSost > 1) {
          AtmMtr[1] = Temp;
          for(NSost1 = 0;NSost1 < AtmMtr[1] -> NSost;++NSost1) {
            Temp = AtmMtr[1] -> Conn[NSost1];
            if ((Temp -> NSost > 1) && (Temp != AtmMtr[1])) {
              AtmMtr[2] = Temp;
              for(NSost2 = 0;NSost2 < AtmMtr[2] -> NSost;++NSost2) {
                Temp = AtmMtr[2] -> Conn[NSost2];
                if ((Temp -> NSost > 1) && (Temp != AtmMtr[1])) {
                  AtmMtr[3] = Temp;
                  if (Temp == Atm) {
                    Ring = 3;
                    goto Esci;
                  }
                  for(NSost3 = 0;NSost3 < AtmMtr[3] -> NSost;++NSost3) {
                    Temp = AtmMtr[3] -> Conn[NSost3];
                    if ((Temp -> NSost > 1) && (Temp != AtmMtr[2])) {
                      AtmMtr[4] = Temp;
                      if (Temp == Atm) {
                        Ring = 4;
                        goto Esci;
                      }
                      for(NSost4 = 0;NSost4 < AtmMtr[4] -> NSost;++NSost4) {
        		Temp = AtmMtr[4] -> Conn[NSost4];
                        if ((Temp -> NSost > 1) && (Temp != AtmMtr[3])) {
                          AtmMtr[5] = Temp;
                          if (Temp == Atm) {
                            Ring = 5;
                            goto Esci;
                          }
                          for(NSost5 = 0;NSost5 < AtmMtr[5] -> NSost;++NSost5) {
                            if (AtmMtr[5] -> Conn[NSost5] == Atm) {
                              Ring = 6;
                              goto Esci;
                            }
                          }
                        }
                      }
                    }
                  }
                }
              }
            }
	  }
	}
      }
      Esci:
      if (Ring) {
        N3    = 0;  /* Intranular nitrogen with 3 sudstituent */
        TotPi = 0;
        for(k = 0; k < Ring; ++k) {
           for(j = 0; j < TOTHUCK; ++j) {
             if ((PiMtr[j].Atom.S == AtmMtr[k] -> Elem.S) &&
                 (PiMtr[j].NSost == AtmMtr[k] -> NSost)) {
               if (j == 1) ++N3;		/* Don't change the matrix order !*/
               TotPi += PiMtr[j].Npi;
               break;
             }
           }


/*
           This line is only for multi-condensed rings and is not yet implemented
           in current release
	   AtmMtr[k] -> Ring += Ring << l;
*/
           if (AtmMtr[k] -> Ring < 0)
             AtmMtr[k] -> Ring = Ring * -1;
           else AtmMtr[k] -> Ring = Ring;
        }

	if ((TotPi == 6) ||
            ((TotPi == 7) && (N3 == 2)) || /* Protonated imidazole       */
            ((TotPi == 7) && (N3 == 1))    /* Protonated purine/pyridine */
           ) {
/*
	  printf("Elettroni Pi: %d\n", TotPi);
*/
	  for(k = 0; k < Ring; ++k)  {

            if (AtmMtr[k] -> Ring > 0) AtmMtr[k] -> Ring *= -1;
/*
	    printf("%d %.4s %d %.3s_%-4.4s\n",
	           AtmMtr[k] -> Ring, AtmMtr[k] -> Name.C, AtmMtr[k] -> NSost,
		   AtmMtr[k] -> ResName.C, AtmMtr[k] -> ResSeq.C);
*/
	  }
	}
      }
    }
    Atm = Atm -> Ptr;
  }
/*
      for(Atm = InizAtm; Atm; Atm = Atm -> Ptr)
      printf("%4.4s %d\n", Atm -> Name.C, Atm -> Ring);
*/
}


/**** Descriptor conversion ****/

VG_BOOL ToDesc(FFPAR *ParFF, FFDESC *Desc, char *Ptr)
{
  VG_BOOL		Ret = TRUE;

  if (Ptr[1] != '-') Desc -> Atom.C[1] = Ptr[1];
  else Desc -> Atom.S  = (VG_WORD)NULL;
  *Desc -> Atom.C = *Ptr;
  if ((Ptr[2] >= '0') && (Ptr[2] <= '9')) {
    Desc -> NSost = Ptr[2] - '0';
    if (((Ptr[3] >= '3') && (Ptr[3] <= '6')) ||
        (Ptr[3] == '9') || (Ptr[3] == '0')) {
      Desc -> Ring = Ptr[3] - '0';
      if ((Ptr[4] == '0') || (Ptr[4] == '1')) Desc -> Arom = Ptr[4] - '0';
      else Ret = FFError(ParFF, MSG_ERR_FFIELD_ILLAROM);
    } else Ret = FFError(ParFF, MSG_ERR_FFIELD_ILLRING);
  } else Ret = FFError(ParFF, MSG_ERR_FFIELD_ILLDONDORD);

  return Ret;
}


/**** Pop the atom types ****/

void FFPop(register ATOMO *Atm, VG_ULONG *FFBuf)
{
  register VG_ULONG     *Ptr = FFBuf;

  while(Atm) {
    Atm -> Pot.L = *Ptr;
    Atm  = Atm -> Ptr;
    ++Ptr;
  } /* End of while */
}


/**** Push the atom types ****/

VG_ULONG *FFPush(register ATOMO *Atm, VG_ULONG TotAtm)
{
  VG_ULONG              *FFBuf;
  register VG_ULONG     *Ptr;

  if ((FFBuf = (VG_ULONG *)Alloca(TotAtm * sizeof(VG_ULONG))) != NULL) {
    Ptr = FFBuf;
    while(Atm) {
      *Ptr = Atm -> Pot.L;
      Atm  = Atm -> Ptr;
      ++Ptr;
    } /* End of while */
  }

  return FFBuf;
}

