
/*************************************************
****    VEGA - Color managment for OpenGL     ****
**** Copyright 1996-2001, Alessandro Pedretti ****
*************************************************/


#ifdef __WIN32__
#  include <windows.h>
#endif

#include <gl\gl.h>
#include <stdio.h>

#ifdef __BORLANDC__
#pragma  hdrstop
#endif

#include "globdef.h"
#include "globvar.h"

#define  __COLORTABLE
#include "gl_global.h"
#include "gl_colors.h"

#define  VG_DEFAULTCOL          VGCOL_MAGENTA


/**** Color a molecule by atom ****/

VG_ULONG GL_ColorByAtm(register ATOMO *Atm)
{
  register VG_ATMCOL       *AtmCol;
  register VG_UBYTE        CurCol;

  register VG_ULONG        Colored = 0;

  while(Atm) {
    if (Atm -> Active) {
      for(AtmCol = AtmColTbl; (AtmCol -> Elem.S) &&
         (Atm -> Elem.S != AtmCol -> Elem.S); ++AtmCol);
      Atm -> ColorID = AtmCol -> ColorID;
      ++Colored;
    }
    Atm = Atm -> Ptr;
  } /* End of while */

  return Colored;
}


/**** Color by chain ****/

VG_ULONG GL_ColorByChain(register ATOMO *Atm)
{
  register VG_UBYTE     ColorID   = VGCOL_RED;
  register VG_UBYTE     PrecChain = Atm -> ChainID;
  register VG_ULONG     Colored   = 0;

  while(Atm) {
    if (Atm -> ChainID != PrecChain) {
      ++ColorID;
      PrecChain = Atm -> ChainID;
      if (ColorID > VGCOL_NUM) ColorID = VGCOL_RED;
    }
    if (Atm -> Active) {
      Atm -> ColorID = ColorID;
      ++Colored;
    }
    Atm = Atm -> Ptr;
  }

  return Colored;
}


/**** Color by charge ****/

VG_ULONG GL_ColorByCharge(register ATOMO *Atm)
{
  struct _ColorRanges {
    float       Min, Max;
    VG_UBYTE    ColorID;
  };
  register struct _ColorRanges  *Ptr;
  register VG_ULONG             Colored       = 0;
  const struct _ColorRanges     ColorRanges[] = {{-0.50, -0.20, VGCOL_BLUE  },
                                                 {-0.20, -0.10, VGCOL_CYAN  },
                                                 {-0.10,  0.10, VGCOL_GREEN },
                                                 { 0.10,  0.20, VGCOL_YELLOW},
                                                 { 0.20,  1.01, VGCOL_RED   },
                                                 { 0.00,  0.00, 0           }
                                                };
  while(Atm) {
    if (Atm -> Active) {
      for(Ptr = (struct _ColorRanges *)ColorRanges; Ptr -> ColorID; ++Ptr) {
        if ((Atm -> Charge >= Ptr -> Min) && (Atm -> Charge < Ptr -> Max)) {
          Atm -> ColorID = Ptr -> ColorID;
          break;
        }
      } /* End of for */
      ++Colored;
    }
    Atm = Atm -> Ptr;
  }

  return Colored;
}


/**** Color by molecule ****/

VG_ULONG GL_ColorByMol(register ATOMO *Atm)
{
  register VG_UBYTE     ColorID = VGCOL_WHITE;
  register VG_ULONG     Colored = 0;

  while(Atm) {
    if (Atm -> Active) {
      Atm -> ColorID = ColorID;
      ++Colored;
    }

    if (Atm -> Flags & VG_ATMF_MOLEND) {
      ++ColorID;
      if (ColorID > VGCOL_NUM) ColorID = VGCOL_WHITE;
    }

    Atm = Atm -> Ptr;
  }

  return Colored;
}


/**** Color a molecule by residue ****/

VG_ULONG GL_ColorByRes(ATOMO *InizAtm)
{
  register ATOMO         *Atm;
  register VG_RESCOL     *Col    = ResColTab;
  register VG_ULONG      Colored = 0;
  register VG_ULONG      ToColor = 0;

  for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
    if (Atm -> Active) {
      Atm -> ColorID = VGCOL_WHITE;
      ++ToColor;
    }
  } /* End of for */

  while ((Col -> ColorID) && (Colored != ToColor)) {
    for(Atm = InizAtm; (Atm) && (Colored != ToColor); Atm = Atm -> Ptr) {
      if (Atm -> Active) {
        if (Col == ResColTab) ++ToColor;
        if (Atm -> ResName.L == Col -> ResName.L) {
          Atm -> ColorID = Col -> ColorID;
          ++Colored;
        }
      }
    } /* End of for */
    ++Col;
  } /* End of while */

  return Colored;
}


/**** Color by segment ****/

VG_ULONG GL_ColorBySeg(register ATOMO *Atm)
{
  register VG_UBYTE     ColorID = VGCOL_WHITE;
  register VG_ULONG     Colored = 0;

  while(Atm) {
    if (Atm -> Active) {
      Atm -> ColorID = ColorID;
      if (Atm -> ResName.L == *(VG_LONG *)"HOH")
        Atm -> ColorID = VGCOL_BLUE;
      else
        Atm -> ColorID = ColorID;
      ++Colored;
    }

    if (Atm -> Flags & VG_ATMF_SEGEND) {
      if (++ColorID == VGCOL_BLUE) ++ColorID;
      if (ColorID > VGCOL_NUM) ColorID = VGCOL_WHITE;
    }

    Atm = Atm -> Ptr;
  }

  return Colored;
}


/**** Color a molecule ****/

VG_ULONG GL_ColorMol(register ATOMO *Atm, register VG_UBYTE ColorID)
{
  register VG_ULONG      Colored = 0;

  if (ColorID > VGCOL_NUM) ColorID = VGCOL_NUM;

  while(Atm) {
    if (Atm -> Active) {
      Atm -> ColorID = ColorID;
      ++Colored;
    }
    Atm = Atm -> Ptr;
  } /* End of while */

  return Colored;
}


/**** Color surface ****/

void GL_ColorSrf(register VG_SURFACE *Dot, VG_UBYTE ColorID)
{
  register VG_COLOR     *Col;

  if (ColorID > VGCOL_NUM) ColorID = VGCOL_NUM;
  Col = &ColorTable[ColorID];

  while(Dot) {
    Dot -> Color[0] = Col -> R;
    Dot -> Color[1] = Col -> G;
    Dot -> Color[2] = Col -> B;
    Dot = Dot -> Ptr;
  }
}

/**** Color surface by atom ****/

void GL_ColorSrfAtm(register VG_SURFACE *Dot, register ATOMO *Atm)
{
  register VG_ATMCOL    *AtmCol;
  VG_UBYTE              R, G, B;

  register VG_ULONG     PrecAtmNum = 0;

  while(Dot) {
    if (Dot -> AtmNum != PrecAtmNum) {

      /**** Search the atom ****/

      while((Dot -> AtmNum != Atm -> Num) && (Atm))
        Atm = Atm -> Ptr;

      /**** Search the color ****/

      for(AtmCol = AtmColTbl; (AtmCol -> Elem.S) &&
       (Atm -> Elem.S != AtmCol -> Elem.S); ++AtmCol);

      R = ColorTable[AtmCol -> ColorID].R;
      G = ColorTable[AtmCol -> ColorID].G;
      B = ColorTable[AtmCol -> ColorID].B;
    }
    Dot -> Color[0] = R;
    Dot -> Color[1] = G;
    Dot -> Color[2] = B;
    PrecAtmNum = Dot -> AtmNum;
    Dot = Dot -> Ptr;
  } /* End of dot loop */
}


/**** Color surface by color ****/

void GL_ColorSrfCol(VG_SURFACE *Dot, VG_ULONG Color)
{
  GL_ColorSrfColRGB(Dot, Color & 0xff, (Color & 0xff00) / 256,
                   (Color & 0xff0000) / 65536);
}


/**** Color surface by color (RGB version) ****/

void GL_ColorSrfColRGB(register VG_SURFACE *Dot, VG_UBYTE R, VG_UBYTE G, VG_UBYTE B)
{
  while(Dot) {
    Dot -> Color[0] = R;
    Dot -> Color[1] = G;
    Dot -> Color[2] = B;
    Dot = Dot -> Ptr;
  } /* End of while */
}


/**** Get the color ID by name ****/

VG_UBYTE GL_GetColorId(register char *ColStr)
{
  register VG_COLOR      *Color;

  for(Color = ColorTable + 1; Color -> Name; ++Color) {
    if (!strcasecmp(Color -> Name, ColStr)) {
      return Color -> ID;
    }
  } /* End of for */

  return VGCOL_NONE;
}


/**** Get the color ID by RGB value ****/

VG_UBYTE GL_GetColorIdRGB(register VG_ULONG Col)
{
  register VG_COLOR      *Color;

  for(Color = ColorTable + 1; Color -> Name; ++Color) {
    if ((VG_ULONG)(Color -> R + (Color -> G << 8) + (Color -> B << 16)) == Col) {
      return Color -> ID;
    }
  } /* End of for */

  return VGCOL_NONE;
}


