
/*************************************************
****          VEGA - OpenGL Utilities         ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/


#ifdef __WIN32__
#  include <windows.h>
#endif

#include <GL\gl.h>
#include <GL\glu.h>
#include <stdio.h>
#include <math.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "gl_global.h"
#include "gl_colors.h"
#include "gl_menu.h"
#include "gl_prefs.h"

/**** Cylinder types ****/

#define  VG_CYLTYPE_SOLID       1
#define  VG_CYLTYPE_WIRE        2
#define  VG_CYLTYPE_DISC        4

#define  VG_ARM_LEN             0.5

static GLUquadricObj   *SphereQuadric[3];
static GLuint          SphereList[3];

static GLenum          SphereStyles[] = { GLU_FILL, GLU_LINE, GLU_POINT };


/**** Calculate the range of residue number ****/

void GL_CalcResRange(ATOMO *Atm, VG_ULONG *Min, VG_ULONG *Max)
{
  VG_LONG       PrecResSeq = 0;
  VG_ULONG      Temp;

  *Max = 0;
  *Min = 0xefffffff;
  while(Atm) {
    if (Atm -> ResSeq.L  != PrecResSeq) {
      PrecResSeq  = Atm -> ResSeq.L;
      Temp        = Char2Int(Atm -> ResSeq.C, 4);
      if (Temp > *Max) *Max = Temp;
      if (Temp < *Min) *Min = Temp;
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Init spheres ****/

void GL_BuildSpheres(int Res)
{
  register int          k;

  for(k = 0; k < 3; ++k) {
    SphereList[k]    = glGenLists(1);
    SphereQuadric[k] = gluNewQuadric();

    gluQuadricDrawStyle(SphereQuadric[k], SphereStyles[k]);
    glNewList(SphereList[k], GL_COMPILE);
    gluSphere(SphereQuadric[k], 1.0, Res, Res);
    glEndList();
  } /* End of for */
}


/**** Draw a cylinder ****/

void GL_Cylinder(float *end, float *start, int rod_res, float rod_radius,
                 float rod_top_radius, int Mode)
{
  float                 R, RXY, phi, theta, lenaxis[3];
  GLUquadricObj         *objQuadric;

  if (Mode & VG_CYLTYPE_SOLID)
    objQuadric = SphereQuadric[0];
  else
    objQuadric = SphereQuadric[1];

  lenaxis[0] = end[0] - start[0];
  lenaxis[1] = end[1] - start[1];
  lenaxis[2] = end[2] - start[2];

  R = lenaxis[0]*lenaxis[0]+lenaxis[1]*lenaxis[1]+lenaxis[2]*lenaxis[2];

  if (R <= 0.0) return;

  R = sqrt(R);

  /**** Determine phi rotation angle, amount to rotate about Y ****/

  phi = acos(lenaxis[2] / R);

  /**** Determine theta rotation, amount to rotate about Z ****/

  RXY = sqrt(lenaxis[0]*lenaxis[0]+lenaxis[1]*lenaxis[1]);
  if(RXY <= 0.0) {
    theta = 0.0;
  } else {
    theta = acos(lenaxis[0] / RXY);
    if (lenaxis[1] < 0.0) theta = (float) (2.0 * PI) - theta;
  }

  glPushMatrix();
  glTranslatef((GLfloat)(start[0]), (GLfloat)(start[1]), (GLfloat)(start[2]));

  if(theta != 0.0)
    glRotatef((GLfloat) ((theta / PI) * 180.0), 0.0, 0.0, 1.0);

  if(phi != 0.0)
    glRotatef((GLfloat) ((phi / PI) * 180.0), 0.0, 1.0, 0.0);

  gluCylinder(objQuadric, (GLdouble)rod_radius, (GLdouble)rod_top_radius,
	      (GLdouble)R, (GLint)rod_res, 1);

  /**** Draw a disc if requested ****/

  if (Mode & VG_CYLTYPE_DISC) {
    gluQuadricOrientation(objQuadric, (GLenum)GLU_INSIDE);
    gluDisk(objQuadric, (GLdouble)rod_top_radius, (GLdouble)rod_radius,
	  (GLint)rod_res, 1);
    gluQuadricOrientation(objQuadric, (GLenum)GLU_OUTSIDE);
  }

  glPopMatrix();
}


/**** Draw a single atom ****/

void GL_DrawAtm(ATOMO *Atm)
{
  GL_SetColor(Atm -> ColorID);
  glBegin(GL_LINES);
  glVertex3f(Atm -> x - VG_ARM_LEN, Atm -> y, Atm -> z);
  glVertex3f(Atm -> x + VG_ARM_LEN, Atm -> y, Atm -> z);
  glEnd();
  glBegin(GL_LINES);
  glVertex3f(Atm -> x, Atm -> y - VG_ARM_LEN, Atm -> z);
  glVertex3f(Atm -> x, Atm -> y + VG_ARM_LEN, Atm -> z);
  glEnd();
  glBegin(GL_LINES);
  glVertex3f(Atm -> x, Atm -> y, Atm -> z - VG_ARM_LEN);
  glVertex3f(Atm -> x, Atm -> y, Atm -> z + VG_ARM_LEN);
  glEnd();
}


/**** Draw atom labels ****/

void GL_DrawAtmLbl(ATOMO *Atm)
{
  glPushMatrix();
  if ((Atm -> Label == VG_ATMLBL_TYPE) && (*Atm -> Pot.C == '?'))
    GL_SetColor(VGCOL_RED);
  else
    GL_SetColor(VGCOL_WHITE);

  glRasterPos3f(Atm -> x, Atm -> y, Atm -> z);

  switch(Atm -> Label) {
  case VG_ATMLBL_NAME:       /* Atom name    */
    GL_Print("%.4s", Atm -> Name.C);
    break;

  case VG_ATMLBL_ELEMENT:    /* Element name */
    GL_Print("%.4s", Atm -> Elem.C);
    break;

  case VG_ATMLBL_NUMBER:     /* Atom number  */
    GL_Print("%d", Atm -> Num);
    break;

  case VG_ATMLBL_TYPE:       /* Atom type    */
    GL_Print("%.4s", Atm -> Pot.C);
    break;

  case VG_ATMLBL_CHARGE:     /* Atom charge  */
    GL_Print("%.2f", Atm -> Charge);
    break;
  } /* End of switch */
  glPopMatrix();
}


/**** Draw CPK & Liquorice ****/

void GL_DrawCpk(register ATOMO *Atm, VG_UWORD Mode)
{
  int                   CylDrawMode;
  int                   SphereType;
  float                 SphereRad, CylinderRad;
  register ATOMO        **ConnAtm;
  register VG_ULONG     k;
  VG_UWORD              CylRes;
  XYZ                   Mid;

  if (Mode == VG_CPKMOD_WIRE) {
    CylDrawMode = VG_CYLTYPE_WIRE;
    SphereType  = SphereList[1];
  } else {
    CylDrawMode = VG_CYLTYPE_SOLID;
    SphereType  = SphereList[0];
  }

  glLineWidth(1.0);

  if (Mode > VG_CPKMOD_WIRE) {
    SphereRad = CylinderRad = GlPrefs -> GlLiqCylRad;
    CylRes    = GlPrefs -> GlLiqCylRes;
  } else {
    CylinderRad = GlPrefs -> GlCpkCylRad;
    CylRes      = GlPrefs -> GlCpkCylRes;
  }

  while(Atm) {
    if ((Mode == VG_CPKMOD_SOLID) || (Mode == VG_CPKMOD_WIRE))
      SphereRad   = Atm -> Rad * GlPrefs -> GlCpkScale;

    if (!Atm -> Active) {
      Atm = Atm -> Ptr;
      continue;
    }

    ++ViewPrefs.RenderedAtoms;

    if (Atm -> Label != VG_ATMLBL_NONE) GL_DrawAtmLbl(Atm);

    glPushMatrix();
      GL_SetColor(Atm -> ColorID);
      glTranslatef(Atm -> x, Atm -> y, Atm -> z);
      glScalef(SphereRad, SphereRad, SphereRad);
      glCallList(SphereType);
    glPopMatrix();

    if (Atm -> NSost) {
      ConnAtm = Atm -> Conn;
      for(k = 0; k < (VG_ULONG)Atm -> NSost; ++k) {
        if (((*ConnAtm) -> Active) && ((*ConnAtm) -> Num > Atm -> Num)) {
          GL_SetColor(Atm -> ColorID);
          if (Atm -> ColorID != (*ConnAtm) -> ColorID) {
            Mid.x = ((*ConnAtm) -> x + Atm -> x) / 2.0;
            Mid.y = ((*ConnAtm) -> y + Atm -> y) / 2.0;
            Mid.z = ((*ConnAtm) -> z + Atm -> z) / 2.0;
            GL_Cylinder(&Mid.x, &Atm -> x, CylRes, CylinderRad, CylinderRad, CylDrawMode);
            GL_SetColor((*ConnAtm) -> ColorID);
            GL_Cylinder(&(*ConnAtm) -> x, &Mid.x, CylRes, CylinderRad, CylinderRad, CylDrawMode);
          } else {
            GL_Cylinder(&(*ConnAtm) -> x, &Atm -> x, CylRes, CylinderRad, CylinderRad, CylDrawMode);
          }
        }
        ++ConnAtm;
      } /* End of connected atoms for */
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}


/**** Draw in wireframe mode ****/

void GL_DrawWireframe(register ATOMO *Atm)
{
  register ATOMO        **ConnAtm;
  register VG_ULONG     k;
  register float        MidX, MidY, MidZ;
  VG_BOOL               Conn;

  glLineWidth(GlPrefs -> GlWireTick);

  while(Atm) {
    if (Atm -> Active) {
      ++ViewPrefs.RenderedAtoms;
      if (Atm -> Label != VG_ATMLBL_NONE) GL_DrawAtmLbl(Atm);
      if (Atm -> NSost) {
        glBegin(GL_LINES);
          Conn    = FALSE;
          ConnAtm = Atm -> Conn;
          for(k = 0; k < (VG_ULONG)Atm -> NSost; ++k) {
            if ((*ConnAtm) -> Active) {
              Conn = TRUE;
              if ((*ConnAtm) -> Num > Atm -> Num) {
                GL_SetColor(Atm -> ColorID);
                glVertex3f(Atm -> x, Atm -> y, Atm -> z);
                if (GlPrefs -> GlSmoothWire) {
                  GL_SetColor((*ConnAtm) -> ColorID);
                  glVertex3f((*ConnAtm) -> x, (*ConnAtm) -> y, (*ConnAtm) -> z);
                } else {
                  if (Atm -> ColorID != (*ConnAtm) -> ColorID) {
                    MidX = ((*ConnAtm) -> x + Atm -> x) / 2.0;
                    MidY = ((*ConnAtm) -> y + Atm -> y) / 2.0;
                    MidZ = ((*ConnAtm) -> z + Atm -> z) / 2.0;
                    glVertex3f(MidX, MidY, MidZ);
                    GL_SetColor((*ConnAtm) -> ColorID);
                    glVertex3f(MidX, MidY, MidZ);
                  }
                  glVertex3f((*ConnAtm) -> x, (*ConnAtm) -> y, (*ConnAtm) -> z);
                }
              }
            }
            ++ConnAtm;
          } /* End of for */
        glEnd();
        if (!Conn) GL_DrawAtm(Atm);
      } else {
        GL_DrawAtm(Atm);
      }
    }
    Atm = Atm -> Ptr;
  } /* End of while */
}

/**** Draw Van der Waals spheres ****/

void GL_DrawVdw(register ATOMO *Atm, VG_UWORD Mode)
{
  glLineWidth(1.0);
  glPointSize(GlPrefs -> GlVdwDotSize);

  while(Atm) {
    if (Atm -> Active) {
      ++ViewPrefs.RenderedAtoms;
      if (Atm -> Label != VG_ATMLBL_NONE) GL_DrawAtmLbl(Atm);
      glPushMatrix();
      GL_SetColor(Atm -> ColorID);
      glTranslatef(Atm -> x, Atm -> y, Atm -> z);
      glScalef(Atm -> Rad, Atm -> Rad, Atm -> Rad);
      glCallList(SphereList[Mode]);
      glPopMatrix();
    }
    Atm = Atm -> Ptr;
  } /* End of while */
  glFlush();
}


/**** Remove spheres ****/

void GL_KillSpheres(void)
{
  register int          k;

  for(k = 0; k < 3; ++k) {
   glDeleteLists(SphereList[k], 1);
   gluDeleteQuadric(SphereQuadric[k]);
  } /* End of for */
}


/**** Reset all memory data ****/

void GL_Reset(VG_GLPREFS *Prf)
{
  CloseAllMol();                         /* Free the memory          */
  Prf -> DrawMode    = VG_GLDRAW_WIREFRAME;
  Prf -> Center.x    = 0.0;              /* Reset rotation center    */
  Prf -> Center.y    = 0.0;
  Prf -> Center.z    = 0.0;
  Prf -> Animation   = FALSE;            /* Stop animation           */
  Prf -> Light       = TRUE;             /* Activate the light       */
  Prf -> TotMol      = 0;
  Prf -> TxtStr      = NULL;             /* Disable text             */
  Prf -> SrfAct      = FALSE;            /* Disable surface          */
  Prf -> MouseMode   = VG_MOUSE_ROT;     /* Mouse mode rotation      */
  Prf -> MeasureMode = VG_PM_PICKNONE;   /* Measure mode             */
  GL_ResetMtx(Prf);

  ChangeMousePopup();
  ChangePickPopup();
}


/**** Reset the transformation matrix ****/

void GL_ResetMtx(VG_GLPREFS *Prf)
{
#ifdef WIN32
  ZeroMemory(RotMat, 16 * sizeof(GLfloat));
#else
  memset(RotMat, 0, 16 * sizeof(GLfloat));
#endif
  RotMat[0][0] = 1.0;
  RotMat[1][1] = 1.0;
  RotMat[2][2] = 1.0;
  RotMat[3][3] = 1.0;

  Prf -> Scale       = 1.0;              /* Reset scale              */
  Prf -> StepRot.x   = 0.0;              /* Reset animation steps    */
  Prf -> StepRot.y   = VG_GLDEF_ROTSTEP;
  Prf -> StepRot.z   = 0.0;
  Prf -> StepTrans.x = 0.0;              /* Reset translations steps */
  Prf -> StepTrans.y = 0.0;
  Prf -> StepTrans.z = 0.0;
}


/**** Reset the view ****/

void GL_ResetView(VG_GLPREFS *Prf)
{
  Prf -> Center      = LastCent;         /* Reset rotation center    */
  Prf -> Animation   = FALSE;            /* Stop animation           */
  Prf -> Light       = TRUE;             /* Activate the light       */
  GL_ResetMtx(Prf);

  ChangeAnimMenu(FALSE);
  ChangeLightMenu(TRUE);
}


/**** Rotate the transformation matrix ****/

void GL_RotXYZ(int a, int b, int c, double Theta, GLfloat A[4][4])
{
  register GLfloat      ct, st, T1, T2, T3;

  ct = cos(Theta);
  st = sin(Theta);

  T1 = A[0][0]*(a+ct*(1-a)) + A[0][1]*(  st*c)     + A[0][2]*( -st*b);
  T2 = A[0][0]*( -st*c)     + A[0][1]*(b+ct*(1-b)) + A[0][2]*(  st*a);
  T3 = A[0][0]*(  st*b)     + A[0][1]*( -st*a)     + A[0][2]*(c+ct*(1-c));
  A[0][0] = T1;
  A[0][1] = T2;
  A[0][2] = T3;

  T1 = A[1][0]*(a+ct*(1-a)) + A[1][1]*(  st*c)     + A[1][2]*( -st*b);
  T2 = A[1][0]*( -st*c)     + A[1][1]*(b+ct*(1-b)) + A[1][2]*(  st*a);
  T3 = A[1][0]*(  st*b)     + A[1][1]*( -st*a)     + A[1][2]*(c+ct*(1-c));
  A[1][0] = T1;
  A[1][1] = T2;
  A[1][2] = T3;

  T1 = A[2][0]*(a+ct*(1-a)) + A[2][1]*(  st*c)     + A[2][2]*( -st*b);
  T2 = A[2][0]*( -st*c)     + A[2][1]*(b+ct*(1-b)) + A[2][2]*(  st*a);
  T3 = A[2][0]*(  st*b)     + A[2][1]*( -st*a)     + A[2][2]*(c+ct*(1-c));
  A[2][0] = T1;
  A[2][1] = T2;
  A[2][2] = T3;

  ViewPrefs.Render = TRUE;
}


/**** Set the atom labels ****/

void GL_SetAtmLbl(register ATOMO *Atm, register VG_UBYTE Label)
{
  while(Atm) {
    if (Atm -> Active)
      Atm -> Label = Label;
    Atm = Atm -> Ptr;
  } /* Endo of while */
}


/**** Change the endian of the rotation matrix ****/

#ifdef LITTLE_ENDIAN
void GL_SwapRotMat(void)
{
  register int  a, b;

  for(a = 0; a < 4; ++a) {
    for(b = 0; b < 4; ++b) {
      Swap(&RotMat[a][b]);
    } /* End of for */
  } /* End of for */
}
#endif


/**** Ribbon rendering ****/

void GL_DrawRibbon(register ATOMO *Atm, VG_UWORD Mode)
{
  GLfloat       t, x, y;
  GLfloat       *Cos_t, *Sin_t;
  VG_ULONG      i;

  VG_ULONG      SplineRes = 5;
  VG_ULONG      RibbonRes = 5;

  VG_ULONG      NumSeg    = 1 + 2 * RibbonRes;

  if (((Cos_t = (GLfloat *)Alloca(NumSeg)) != NULL) &&
      ((Sin_t = (GLfloat *)Alloca(NumSeg)) != NULL)) {
    Cos_t[0] = 1.0f;
    Sin_t[0] = 0.0f;
    x        = (3.141592653589793238f) / RibbonRes;

    for(i = 1;i < RibbonRes;i++) {
      t                    = x*i;
      Cos_t[i]             = (GLfloat)(cos(t));
      Sin_t[i]             = -(GLfloat)(sin(t));
      Cos_t[i + RibbonRes] = -Cos_t[i];
      Sin_t[i + RibbonRes] = -Sin_t[i];
    } /* End of for */

    Cos_t[RibbonRes]  = -1.0f;
    Sin_t[RibbonRes]  = 0.0f;
    Cos_t[NumSeg - 1] = 1.0f;
    Sin_t[NumSeg - 1] = 0.0f;


    FREE(Cos_t);
    FREE(Sin_t);
  }
}


