
/*****************************************************
****                 VEGA - ILM 1.0               ****
**** Calculate the Molecular Hydropathicity Index ****
****           with G. Vistoli method.            ****
****   Copyright 1996-2002, Alessandro Pedretti   ****
*****************************************************/


#include <stdio.h>
#include <math.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "formats.h"

/**** Local prototypes ****/

static void CalcBarWat(XYZ *, ATOMO **);


/**** Calculate the mass center of water molecule ****/

static void CalcBarWat(register XYZ *Vect, register ATOMO **Atm)
{
  register float         Ms;
  register VG_ULONG      k;

  Vect -> x = Vect -> y = Vect -> z = 0.0;

  for(k = 0;(k < 3) && (*Atm); ++k) {
#ifdef LITTLE_ENDIAN
    if (((ATOMO *)*Atm) -> Elem.S == 0x48)
#else
    if (((ATOMO *)*Atm) -> Elem.S == 0x4800)
#endif
      Ms = 1.0;
    else Ms = 16.0;

    Vect -> x += ((ATOMO *)*Atm) -> x * Ms;
    Vect -> y += ((ATOMO *)*Atm) -> y * Ms;
    Vect -> z += ((ATOMO *)*Atm) -> z * Ms;

    *Atm = ((ATOMO *)*Atm) -> Ptr;
  }
  Vect -> x /= 18.0;
  Vect -> y /= 18.0;
  Vect -> z /= 18.0;
}


/**** Calculate the cut-off value ****/

float IlmClacCutOff(ATOMO *InizAtm)
{
  ATOMO                 *Atm1, *Atm2;
  register float         Dist;
  XYZ                    Vect1, Vect2;

  float                  MaxDist = 0.0;

  Atm1 = InizAtm;
  while(Atm1) {
#ifdef LITTLE_ENDIAN
    if (Atm1 -> ResName.L == 0x484F48) {
#else
    if (Atm1 -> ResName.L == 0x00484F48) {
#endif
      CalcBarWat(&Vect1, &Atm1);
      Atm2 = InizAtm;
      while(Atm2) {
#ifdef LITTLE_ENDIAN
        if (Atm2 -> ResName.L == 0x484F48) {
#else
        if (Atm2 -> ResName.L == 0x00484F48) {
#endif
          CalcBarWat(&Vect2, &Atm2);
          Dist = Quad(Vect1.x - Vect2.x) + Quad(Vect1.y - Vect2.y) + Quad(Vect1.z - Vect2.z);
          if (Dist > MaxDist) MaxDist = Dist;
        } else Atm2 = Atm2 -> Ptr;
      } /* End of Atm2 loop */
    } else Atm1 = Atm1 -> Ptr;
  } /* End of Atm1 loop */

  return SQR(MaxDist) / 2.0;
}


/**** Calculate the ILM value fon an entire molecule ****/

float CalcIlm(ATOMO *Atm, float QuadCutOff, VG_ULONG Flags)
{
  ATOMO         *Wat;
  double        AtmIlm, Dist;
  XYZ           Vect;
  VG_ULONG      NWat;

  ATOMO         *BegWat  = Atm;
  double        Ilm      = 0.0;
  VG_BOOL       FirstWat = FALSE;
  VG_ULONG      NAtm     = 0;


  while(Atm) {
    if ((Flags & SRF_FLG_SELONLY) && (!Atm -> Active)) continue;

#ifdef LITTLE_ENDIAN
    if (Atm -> ResName.L != 0x484F48) {
#else
    if (Atm -> ResName.L != 0x00484F48) {
#endif
      AtmIlm = 0.0;
      NWat   = 0;
      Wat    = BegWat;
      do {
#ifdef LITTLE_ENDIAN
        if (Wat -> ResName.L == 0x484F48) {
#else
        if (Wat -> ResName.L == 0x00484F48) {
#endif
          if (!FirstWat) {
            BegWat   = Wat;
            FirstWat = TRUE;
          }
          CalcBarWat(&Vect, &Wat);
          Dist = Quad(Atm -> x - Vect.x) + Quad(Atm -> y - Vect.y) + Quad(Atm -> z - Vect.z);
          if (Dist < QuadCutOff) {
            AtmIlm += (double)SQR(Dist);
            ++NWat;
          }
        } else Wat  = Wat -> Ptr;
      } while(Wat);
    }

    if (NWat) {
      Ilm += AtmIlm / NWat;
      ++NAtm;
    }

    Atm = Atm -> Ptr;
  } /* End of while */

  return (float)(Ilm / NAtm);
}


/**** Calculate a single point ILM ****/

float CalcSrfIlm(ATOMO *Atm, float X, float Y, float Z, float CutOff)
{
  register double        Dist;
  XYZ                    Vect;

  register float         Ilm  = 0.0;
  register VG_ULONG      NWat = 0;

  CutOff *= CutOff;

  do {
#ifdef LITTLE_ENDIAN
    if (Atm -> ResName.L == 0x484F48) {
#else
    if (Atm -> ResName.L == 0x00484F48) {
#endif
      CalcBarWat(&Vect, &Atm);
      Dist = Quad(X - Vect.x) + Quad(Y - Vect.y) + Quad(Z - Vect.z);
      if (Dist < CutOff) {
        Ilm += (double)SQR(Dist);
        ++NWat;
      }
    } else Atm  = Atm -> Ptr;
  } while(Atm);

  if (NWat) Ilm /= (float)NWat;

  return Ilm;
}




