
/*************************************************
****       VEGA - Localization routines       ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/

/*
 * The following routines use the AmigaOS Locale Technology to translate the
 * strings in any language using a specific catalog file in standard IFF format.
 * Each locale file must be stored in a sub-directory with the lower-case language
 * name of the Locale directory and must have VEGA.catalog file name.
 * To create a new catalog, you must use FlexCat compiler with the catalog
 * description file (.cd) and the catalog translation file (.ct). Please read the
 * specific documentation in FlexCat directory and in VEGA manual.
 * The AmigaOS Locale Technology is copyrighted by Amiga Inc.
 */


#ifdef __WIN32__
#  include <windows.h>
#endif

#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#define  VEGA_ARRAY
#include "globstr.h"
#define  __VG_ERRTRANSTAB
#include "comerr.h"
#include "../LocaleLib/locale.h"


#ifdef AMIGA
#  include <exec/libraries.h>
#  include <libraries/locale.h>

extern struct Library  *LocaleBase;
#endif

#ifdef WIN32
#include "gl_global.h"
#include "winutils.h"

extern HWND             hWnd;
static char             LocBuf[4096];

/**** Library function pointers ****/

void           (WINAPI *VG_LOCALE_LocCloseCat)(void *)                                      = NULL;
char          *(WINAPI *VG_LOCALE_LocGetCatStr)(void *, unsigned int, char *)               = NULL;
int            (WINAPI *VG_LOCALE_LocGetErr)(void)                                          = NULL;
void           (WINAPI *VG_LOCALE_LocGetErrStr)(char *, int)                                = NULL;
char          *(WINAPI *VG_LOCALE_LocGetLangStr)(void)                                      = NULL;
void           (WINAPI *VG_LOCALE_LocIso2Dos)(char *, short)                                = NULL;
void          *(WINAPI *VG_LOCALE_LocOpenCat)(char *, char *, char *, unsigned int, char *) = NULL;

/**** Library definitions ****/

#define  LocCloseCat            VG_LOCALE_LocCloseCat
#define  LocGetCatStr           VG_LOCALE_LocGetCatStr
#define  LocGetErr              VG_LOCALE_LocGetErr
#define  LocGetErrStr           VG_LOCALE_LocGetErrStr
#define  LocGetLangStr          VG_LOCALE_LocGetLangStr
#define  LocIso2Dos             VG_LOCALE_LocIso2Dos
#define  LocOpenCat             VG_LOCALE_LocOpenCat

static HINSTANCE                hLocale;

int     ConPrintf(FILE *FH, char *, va_list);
#endif


/**** Close the catalog ****/

void CatClose(void)
{
#ifdef AMIGA
  if (LocaleBase) {
    CloseCatalog(Catalog);
    CloseLibrary(LocaleBase);
  }
#else
#  ifdef WIN32
  if (hLocale) {
    LocCloseCat(Catalog);
    FreeLibrary(hLocale);
  }
#  else
  LocCloseCat(Catalog);
#  endif
#endif
}


/**** Shows an error ****/

VG_BOOL CatErr(VG_ULONG Err, ...)
{
  char                  *Ptr;
  va_list               vl;

  SetGlobErr(Err);
  va_start(vl, Err);
  vsprintf(ErrStr, GetStr(Err), vl);
  va_end(vl);

  switch (ErrMode) {
  case VG_ERRMOD_WINDOW:
#ifdef WIN32
    if (GLOBSW_OPENGL) {
      sprintf(LocBuf, GetStr(MSG_ERR_LOCALE_ERRCODE), GlobErr, ErrStr);
      GL_MessageBox(LocBuf, GetStr(MSG_ERR_LOCALE_ERROR), MB_OK|MB_ICONERROR);
      break;
    }
#endif

  case VG_ERRMOD_CONSOLE:
    LocPrintf(stderr, ERROR_PROMPT, GetStr(MSG_ERR_LOCALE_ERROR), GlobErr, ErrStr);
    break;
  } /* End of switch */

  return FALSE;
}


/**** Initialize the localization ****/

VG_BOOL CatInit(void)
{
#if !defined(AMIGA) && !defined(WIN32)
  char          LocBuf[256];
#endif

#ifdef AMIGA

  /**** Open the locale.library (only for AmigaOS) ****/
  /**** and the language catalog for translation   ****/

  if ((LocaleBase = (struct Library *)OpenLibrary("locale.library", 38))) {
    Catalog  = (void *)OpenCatalog(NULL, "VEGA.catalog", OC_BuiltInLanguage,
               "english", OC_Version, VEGA_CAT_VER, TAG_DONE);
  } else return FALSE;
#else

#  ifdef WIN32
  if ((GetProcAddresses(&hLocale, "locale.dll", 7,
                        &VG_LOCALE_LocCloseCat       , "LocCloseCat",
                        &VG_LOCALE_LocGetCatStr      , "LocGetCatStr",
                        &VG_LOCALE_LocGetErr         , "LocGetErr",
                        &VG_LOCALE_LocGetErrStr      , "LocGetErrStr",
                        &VG_LOCALE_LocGetLangStr     , "LocGetLangStr",
                        &VG_LOCALE_LocIso2Dos        , "LocIso2Dos",
                        &VG_LOCALE_LocOpenCat        , "LocOpenCat"
                       ) == FALSE))
    return FALSE;
#  endif

  /**** Open the catalog for language translation ****/

  GetPrgPath(LocBuf, FALSE);

  if ((Catalog = (void *)LocOpenCat("VEGA.catalog",
                                    LocBuf,
                                    Prefs.LANGUAGE,
                                    VEGA_CAT_VER,
                                    "english")) == NULL) {
    if ((LocGetErr()) && (LocGetErr() != LOCERR_NOTFOUND) &&
        (LocGetErr() != LOCERR_AUTOWIN)) {
#  ifndef WIN32
      if (LocGetErr() == LOCERR_AUTONOTSUP) {
        PrintWarn("Your operating system doesn't support the automatic language\n" \
                  "detection. Please set the LANGUAGE key in the \"prefs\" file\n" \
                  "with your preferred language\n");
      } else {
#  endif
        LocGetErrStr(LocBuf, 256);
        GlobErr = VG_PARET_LOCALE;
        Errore(LocBuf);
#  ifndef WIN32
      }
#  endif
    }
  }
#endif

  return TRUE;
}


/**** fprintf replacement with string localization ****/

int CatPrintf(FILE *FH, VG_ULONG Str, ...)
{
  int           Ret;
  va_list       vl;

  va_start(vl, Str);

#ifdef WIN32
  Ret = ConPrintf(FH, GetStr(Str), vl);
#else
  Ret = vfprintf(FH, GetStr(Str), vl);
#endif

  va_end(vl);

  return Ret;
}


/**** fprintf replacement with iso2dos character conversion ****/

#ifdef WIN32
int DosPrintf(FILE *FH, char *Str, ...)
{
  va_list       vl;
  int		Ret;

  va_start(vl, Str);
  Ret = ConPrintf(stdout, Str, vl);
  va_end(vl);

  return Ret;
}
#endif


/**** Shows an error ****/

VG_BOOL Errore(const char *Err, ...)
{
  char          Buf[128];
  va_list       vl;

#ifndef WIN32
  sprintf(Buf, "\n\033[33m%s\033[0m ", GetStr(MSG_ERR_LOCALE_ERROR));
#else
  sprintf(Buf, "\n%s ", GetStr(MSG_ERR_LOCALE_ERROR));
#endif
  va_start(vl, Err);
  strcat(Buf, Err);
  strcat(Buf, ".\n\n");
#ifdef WIN32
  LocIso2Dos(Buf, FALSE);
#endif

  vfprintf(stderr, Buf, vl);
  va_end(vl);

  return FALSE;
}


/**** Return a string and if possible, translate it ****/

char *GetStr(VG_ULONG Id)
{
  char                  *BuiltIn = "Unknown localized string";
  register VG_ULONG     k        = sizeof(VEGA_Str) / sizeof(LOCARRAY);
  register LOCARRAY     *Ptr     = (LOCARRAY *)VEGA_Str;

  while(k--) {
    if (Ptr -> Id == Id) {
      BuiltIn = Ptr -> Str;
      break;
    }
    ++Ptr;
  }

  /**** String localization ****/

  if (Catalog)
#ifdef AMIGA
    return((char *)GetCatalogStr((void *)Catalog, Id, BuiltIn));
#else
    return((char *)LocGetCatStr((void *)Catalog, Id, BuiltIn));
#endif

  return BuiltIn;
}


/**** Sets the global error variable ****/

void SetGlobErr(VG_ULONG Err)
{
  register struct _ErrTab       *ErrPtr;

  for(ErrPtr = (struct _ErrTab *)ErrTab;
      (ErrPtr -> ErrCode != VG_PARET_UNKERR) && (ErrPtr -> StrId != (VG_UWORD)Err);
      ++ErrPtr);
  GlobErr = ErrPtr -> ErrCode;
}


#ifdef WIN32

/**** Reset global error ****/

void ResetGlobErr(void)
{
  GlobErr = VG_PARET_OK;
  *ErrStr = 0;
}


/**** Show the last error ****/

void ShowLastErr(void)
{
  if (GlobErr == VG_PARET_OK)
    strcpy(ErrStr, GetStr(MSG_ERR_LOCALE_NOERROR));

  switch (ErrMode) {
  case VG_ERRMOD_WINDOW:
    if (GLOBSW_OPENGL) {
      sprintf(LocBuf, GetStr(MSG_ERR_LOCALE_ERRCODE), GlobErr, ErrStr);
      GL_MessageBox(LocBuf, GetStr(MSG_ERR_LOCALE_LASTERR), MB_OK|MB_ICONERROR);
      break;
    }

  case VG_ERRMOD_CONSOLE:
    LocPrintf(stderr, ERROR_PROMPT, GetStr(MSG_ERR_LOCALE_LASTERR), GlobErr, ErrStr);
    break;
  } /* End of switch */
}

/**** Show a global system error ****/

VG_BOOL WinErr(void)
{
//  char          *lpMsgBuf;

  GlobErr = VG_PARET_WINDOWS;

//  FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
  FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM,
                NULL,
                GetLastError(),
                MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                ErrStr, 255,
//                (LPTSTR)&lpMsgBuf, 0,
                NULL);

  switch (ErrMode) {
  case VG_ERRMOD_WINDOW:
    if (GLOBSW_OPENGL) {
      sprintf(LocBuf, GetStr(MSG_ERR_LOCALE_ERRCODE), GlobErr, ErrStr);
      GL_MessageBox(LocBuf, GetStr(MSG_ERR_LOCALE_ERROR), MB_OK|MB_ICONERROR);
      break;
    }

  case VG_ERRMOD_CONSOLE:
    LocPrintf(stderr, ERROR_PROMPT, GetStr(MSG_ERR_LOCALE_ERROR), GlobErr, ErrStr);
    break;
  } /* End of switch */

//  LocalFree(lpMsgBuf);

  return FALSE;
}
#endif



