
/*************************************************
****        VEGA - MOL2 loader & saver        ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/


#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"

#ifndef WIN32
#include <unistd.h>
#endif

#define  FIELDLEN           256
#define  TRIPOS_TEMPLATE    "TRIPOS.tem"

#ifdef LITTLE_ENDIAN
#  define  FF_TRIPOS_C2     0x00322e43
#else
#  define  FF_TRIPOS_C2     0x432e3200
#endif


static char	*Hdr_Atom = "@<TRIPOS>ATOM\n";



/**** Assign the Tripos force field ****/

VG_BOOL AssignTriposFF(ATOMO *InizAtm, VG_ULONG TotAtm)
{
  char			Field[FIELDLEN], FieldName[FIELDLEN];
  FILE                  *IN;
  register ATOMO	*Atm;
  VG_QCHAR		*Template, *TmpPtr;
  VG_ULONG		Count;

  VG_BOOL               Ret    = FALSE;

  /**** Load the force field template ****/

  GetPrgPath(FieldName, TRUE);
  strcat(FieldName, TRIPOS_TEMPLATE);
  if ((IN = fopen(FieldName, "r"))) {
    Count = 0;
    while(fgets(Field, FIELDLEN, IN))
      if ((*Field != '\n') && (*Field != ';')) ++Count;
    fclose(IN);
    if (Count) {
      if ((Template = (VG_QCHAR *)Alloca((Count + 1) * sizeof(VG_QCHAR)))) {
        if ((IN = fopen(FieldName, "r"))) {
          TmpPtr = Template;
          while(fgets(Field, FIELDLEN, IN))
            if ((*Field != '\n') && (*Field != ';')) {
              sscanf(Field, "%4s", TmpPtr -> C);
              ++TmpPtr;
            }
          fclose(IN);

          /**** Check atom types ****/

          Count = 0;
          for(Atm = InizAtm; Atm; Atm = Atm -> Ptr) {
            for(TmpPtr = Template; TmpPtr -> L; ++TmpPtr)
              if (TmpPtr -> L == Atm -> Pot.L) {
                ++Count;
                break;
              }
            if (!TmpPtr -> L) break;
          }
          Ret = TRUE;
          if (Count != TotAtm) {

            /**** Assign TRIPOS force field ****/

            PrintProg(MSG_MOL2_ASSIGNPROG);
            AssignFF(FieldName, InizAtm, TotAtm, TRUE);
          }
         } else Ret = PrintDosErr();
        FREE(Template);
      }
    } else CatErr(MSG_ERR_MOL2_ILLTEMP);
  } else CatErr(MSG_ERR_MOL2_NOTFOUND);

  return Ret;
}


/**** Sybyl Mol2 Loader ****/

ATOMO *Mol2Load(FILE *IN, RECORD *Lin, VG_ULONG *TotAtomi)
{
  char			Name[10], ResName[12], ResSeq[10];

  ATOMO			*InizAtm = NULL;
  register ATOMO	*Atm     = NULL;

  while(fgets(Lin -> Line, LINELEN, IN)) {
    if (!strcmp(Lin -> Line, Hdr_Atom)) {
      while((fgets(Lin -> Line, LINELEN, IN)) && (*Lin -> Line != '@')) {
        if ((*Lin -> Line) && (*Lin -> Line != '\n')) {
          if ((Atm = AllocAtm(&InizAtm, TotAtomi))) {
            sscanf(Lin -> Line, "%*s %8s %f %f %f %4s %8s %10s %f",
                   Name, &Atm -> x, &Atm -> y, &Atm -> z,
                   Atm -> Pot.C, ResSeq, ResName, &Atm -> Charge);
            Str2Qchar(&Atm -> Name   , Name   );
            Str2Qchar(&Atm -> ResName, ResName);
      	    Str2Qchar(&Atm -> ResSeq , ResSeq);
            Atm -> Elem.C[0] = toupper(Atm -> Name.C[0]);
            if ((Atm -> Name.C[1]) && (islower(Atm -> Name.C[1])))
              Atm -> Elem.C[1] = Atm -> Name.C[1];
          } else break;
        }
      } /* End of while */
    }
  } /* End of while */

  if (Atm) {
    Atm -> Flags |= VG_ATMF_MOLEND|VG_ATMF_SEGEND;
    LastAtm       = Atm;
  } else CatErr(MSG_ERR_MOL2_CORRUPTED);

  return InizAtm;
}


/**** Sybyl Mol2 saver ****/

VG_BOOL Mol2Save(FILE *OUT, ATOMO *InizAtm, VG_ULONG TotAtm, char *Name)
{
  ATOMO                 *Atm;
  char			*CharTyp, *LogName, *Ptr;
  FILE			*IN;
  VG_QCHAR		Ord1, Ord2, Ord3;
  register VG_UWORD	k;
  time_t		Orario;
  VG_ULONG		Count;

  char                  *DotPtr  = NULL;
  char                  *BegName = Name;
  VG_BOOL		Ret      = TRUE;

#ifdef __VG_OPENGL
  VG_LONG               *Store = NULL;

  if (GLOBSW_OPENGL)
    Store = StoreFF(InizAtm, TotAtm);
#endif

  if ((Ret = AssignTriposFF(InizAtm, TotAtm))) {
    if (*Name) {
      for(Ptr = Name + strlen(Name); Ptr != Name; --Ptr) {
        if ((!DotPtr) && (*Ptr == '.')) {
          DotPtr = Ptr;
          *Ptr   = 0;
        }
#ifdef WIN32
        if (*Ptr == '\\') {
#else
#  ifdef AMIGA
        if ((*Ptr == '/') || (*Ptr == ':')) {
#  else
        if (*Ptr == '/') {
#  endif
#endif
          BegName = Ptr + 1;
          break;
        }
      } /* End of for */
    } else BegName = "Unknown";
#ifndef WIN32
    LogName = getlogin();
#else
    LogName = "User";
#endif
    Orario  = time(0);
    Ptr     = ctime(&Orario);
    if (ChkCharges(InizAtm, TotAtm, NULL)) CharTyp = "USER";
    else CharTyp = "NO";
    if (fprintf(OUT, "#       Name:                   %s\n" \
                     "#       Creating user name:     %s\n" \
                     "#       Creation time:          %s\n\n" \
                     "#       Modifying user name:    %s\n" \
                     "#       Modification time:      %s\n\n" \
                     "@<TRIPOS>MOLECULE\n%s\n" \
                     "%5d %5d    0    0    0\n" \
                     "SMALL\n%s_CHARGES\n\n%s\n" \
                     "@<TRIPOS>ATOM\n",
                      BegName, LogName, Ptr, LogName, Ptr, BegName, TotAtm,
                      CountBond(InizAtm), CharTyp, VEGAHdr) > 0) {
      Count = 1;
      for(Atm = InizAtm; (Atm && Ret); Atm = Atm -> Ptr) {
        if (fprintf(OUT, "%7d %-8.4s %9.4f %9.4f %9.4f %-4.4s %6.4s %-8.4s %9.4f\n",
                         Count++, Atm -> Name.C, Atm -> x, Atm -> y, Atm -> z,
                         Atm -> Pot.C, Atm -> ResSeq.C, Atm -> ResName.C,
                         Atm -> Charge) <= 0) {
          Ret = PrintDosErr();
        }
      }
      if (Ret) {
        if (fprintf(OUT, "@<TRIPOS>BOND\n") > 0) {
          Count = 1;
          for(Atm = InizAtm; (Atm && Ret); Atm = Atm -> Ptr) {
            Ord1.L = AttToOrd(Atm);
            for(k = 0; ((Ret) && (k < Atm -> NSost)); ++k) {
              if (Atm -> Conn[k] -> Num > Atm -> Num) {
                Ord2.L = AttToOrd(Atm -> Conn[k]);
                if ((Ord1.L == 0x61720000) && (Ord2.L != 0x61720000)) Ord3.L = 0x31000000;
                else if ((Ord2.L == 0x61720000) && (Ord1.L != 0x61720000)) Ord3.L = 0x31000000;
                     else if (((Ord2.L == 0x616d0000) && (Atm -> Pot.L == FF_TRIPOS_C2)) ||
                              ((Ord1.L == 0x616d0000) && (Atm -> Conn[k] -> Pot.L == FF_TRIPOS_C2))) Ord3.L = 0x616d0000;
                          else if ((VG_ULONG)(Ord1.L) >= (VG_ULONG)(Ord2.L)) Ord3.L = Ord2.L;
                               else Ord3.L = Ord1.L;
#ifdef LITTLE_ENDIAN
                Swap(&Ord3.L);
#endif
                if (fprintf(OUT, "%6d %4d %4d %.4s\n",
                            Count++, Atm -> Num, Atm -> Conn[k] -> Num, Ord3.C) <= 0)
                  Ret = PrintDosErr();
              }
            } /* End of for */
          } /* End of for */
        } else Ret = PrintDosErr();
      }
    } else Ret = PrintDosErr();
    if (DotPtr) *DotPtr = '.';
  }

#ifdef __VG_OPENGL
  if (Store) {
    RestoreFF(InizAtm, Store);
    FREE(Store);
  }
#endif

  return Ret;
}


/**** Translate atom type to bond order ****/

VG_ULONG AttToOrd(ATOMO *Atm)
{
  VG_ULONG      Att = 0;

  if (!Atm -> NSost) Att = 0x30000000;
  else Att = 0x31000000;

  if (Atm -> Pot.C[1] == '.') {
    if ((Atm -> Pot.C[2] == 'a') && (Atm -> Pot.C[3] == 'r')) {
      Att = 0x61720000;
    } else {
      if ((Atm -> Pot.C[2] == 'a') && (Atm -> Pot.C[3] == 'm')) {
        Att = 0x616d0000;
      } else {
        if (Atm -> Pot.C[2] == '4') Att = 0x31000000;
        else Att = ((VG_ULONG)(0x34 + 0x30 - Atm -> Pot.C[2])) << 24;
      }
    }
  }

  return Att;
}

