
/*************************************************
****                VEGA - PDB                ****
****   Loader, saver and trajectory routines  ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/

/*
 * Supported PDB formats:
 *
 * PDB standard          read & write
 * PDB non standard      read & write
 * PDBQ                  write only
 * PDB Orac              read only
 * PDBF (Fat subformat)  read & write
 * PDBA (Atdl subformat) read & write
 */

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "formats.h"

/**** Local definitions ****/

struct _PdbExt {
  VG_UWORD      Type;
  VG_ULONG      FoundStr;
  char          *RecStr;
};

static const struct _PdbExt     PdbExt[] = {{PDB_ATDL, MSG_PDB_ATDLFOUND, "REMARK  78"},
                                            {PDB_FAT , MSG_PDB_FATFOUND , "REMARK  77 EXTRA"},
                                            {PDB_ORAC, MSG_PDB_ORACFOUND, "REMARK   1 Configuration at time step"},
                                            {0       , 0                , NULL }
                                           };


/**** Close the trajectory file ****/

void PDBClose(TRJINFO *Info)
{
  if (Info -> FH)      fclose(Info -> FH);
  if (Info -> SeekTab) FREE(Info -> SeekTab);
}


/**** Universal PDB Loader ****/

ATOMO *PDBLoad(FILE *PDBIN, RECORD *PDBLin, VG_ULONG *TotAtomi, VG_BOOL *IsMulti)
{
  char                          *j, ResSeq[20];
  register struct _PdbExt       *Ptr;
  VG_LONG                       k;
  VG_UWORD                      l;

  ATOMO                         *InizAtm  = NULL;
  ATOMO                         *PrecAtm  = NULL;
  register ATOMO                *Atm      = NULL;
  VG_BOOL                       FirstAtm  = FALSE;
  VG_UWORD                      IsPdbf    = 0;

  fseek(PDBIN, 0, SEEK_SET);
  while(fgets(PDBLin -> Line, LINELEN, PDBIN)) {

    /**** PDBA/PDBF recognition ****/

    if (PDBLin -> Hdr == *(VG_LONG *)"REMA") {
      for(Ptr = (struct _PdbExt *)PdbExt; Ptr -> RecStr; ++Ptr) {
        if (!strncmp(PDBLin -> Line, Ptr -> RecStr, strlen(Ptr -> RecStr))) {
          if (!IsPdbf)
            PrintProg(Ptr -> FoundStr);
          IsPdbf = Ptr -> Type;
          break;
        }
      } /* End of for */

      switch(IsPdbf) {
      case PDB_ATDL:
        if ((Atm = AllocAtm(&InizAtm, TotAtomi))) {
      	  sscanf(PDBLin -> Line + 16, "%f %4s", &Atm -> Charge, Atm -> Pot.C);
        } else break;
        break;

      case PDB_FAT:
        if ((Atm = AllocAtm(&InizAtm, TotAtomi))) {
      	  sscanf(PDBLin -> Line + 26, "%4s %f", Atm -> Pot.C, &Atm -> Charge);
        } else break;
        break;
      } /* End of switch */
    } else if ((PDBLin -> Hdr == *(VG_LONG *)PDBAtmRec[0]) ||
               (PDBLin -> Hdr == *(VG_LONG *)PDBAtmRec[1])) {
      if (((IsPdbf == PDB_ATDL) || (IsPdbf == PDB_FAT)) && (!FirstAtm)) {
        Atm      = InizAtm;
        PrecAtm  = InizAtm;
        FirstAtm = TRUE;
      }
      if ((IsPdbf != PDB_ATDL) && (IsPdbf != PDB_FAT))
        Atm = AllocAtm(&InizAtm, TotAtomi);
      if (Atm) {
      	sscanf(PDBLin -> Line + 12, "%4s", Atm -> Name.C);
        if (IsPdbf != PDB_ORAC) {
      	  l = *Atm -> Name.C;
      	  if (isdigit(l)) {
            for(k = 0;(k < 4) && (Atm -> Name.C[k]); ++k);
            for(j = Atm -> Name.C;j < (Atm -> Name.C + sizeof(Atm -> Name.C)); ++j)
              *j = j[1];
            Atm -> Name.C[--k] = l;
      	  }
      	  *Atm -> Elem.C = toupper(*Atm -> Name.C);
      	  if ((isalpha(PDBLin -> Line[12])) && (!isdigit(Atm -> Name.C[1])))
      	    Atm -> Elem.C[1] = tolower(Atm -> Name.C[1]);
        } else *Atm -> Elem.C = *Atm -> Name.C;
        Atm -> ResName.L = 0;
      	sscanf(PDBLin -> Line + 17, "%3s", Atm -> ResName.C);
      	Atm -> ChainID = PDBLin -> Line[21];
      	sscanf(PDBLin -> Line + 22, "%4s", ResSeq);
      	Str2Qchar(&Atm -> ResSeq, ResSeq);
      	sscanf(PDBLin -> Line + 30, "%8f%8f%8f", &Atm -> x, &Atm -> y, &Atm -> z);
      	if (PDBLin -> Hdr == *(VG_LONG *)PDBAtmRec[0])
          Atm -> Flags = VG_ATMF_NONE;
      	else
          Atm -> Flags = VG_ATMF_HETATM;
        PrecAtm = Atm;
        if ((IsPdbf == PDB_ATDL) || (IsPdbf == PDB_FAT))
          Atm = Atm -> Ptr;
      } else break;
    } else if (PDBLin -> Hdr == *(VG_LONG *)"TER ") {
      PrecAtm -> Flags |= VG_ATMF_SEGEND;
      if (IsPdbf == PDB_ORAC) break;
    } if (PDBLin -> Hdr == *(VG_LONG *)"ENDM") {
      if (IsMulti) *IsMulti = TRUE;
      break;
    }
  } /* End of while */

  if (PrecAtm) PrecAtm -> Flags |= VG_ATMF_MOLEND|VG_ATMF_SEGEND;
  LastAtm = PrecAtm;

  return InizAtm;
}


/**** Open the trajectory file ****/

VG_BOOL PDBOpen(char *FileName, TRJINFO *Info)
{
  RECORD        PDBLin;
  VG_ULONG      *Ptr;

  VG_BOOL       Ret = TRUE;

  if ((Info -> FH = fopen(FileName, "rb")) != NULL) {

    /**** Count the number of frames ****/

    while(fgets(PDBLin.Line, LINELEN, Info -> FH)) {
      if (PDBLin.Hdr == *(VG_LONG *)"MODE")
        ++Info -> Frames;
    } /* End of while */

    if (Info -> Frames) {
      fseek(Info -> FH, 0, SEEK_SET);

      /**** Create the seek table ****/

      if ((Info -> SeekTab = (VG_ULONG *)Alloca(Info -> Frames * sizeof(VG_ULONG))) != NULL) {
        Ptr = Info -> SeekTab;
        while(fgets(PDBLin.Line, LINELEN, Info -> FH)) {
          if (PDBLin.Hdr == *(VG_LONG *)"MODE") {
            *Ptr++ = ftell(Info -> FH);
          }
        } /* End of while */
        Info -> MolAtm = TotalAtm;
      } else Ret = FALSE;
    }
  } else Ret = PrintDosErr();

  if (!Ret) PDBClose(Info);

  return Ret;
}


/**** Read one trajectory frame ****/

VG_BOOL PDBReadFrm(TRJINFO *Info, ATOMO *Atm)
{
  RECORD        PDBLin;

  VG_ULONG      Num = 0;

  while((fgets(PDBLin.Line, LINELEN, Info -> FH)) &&
        (PDBLin.Hdr != *(VG_LONG *)"ENDM")) {
    if ((PDBLin.Hdr == *(VG_LONG *)PDBAtmRec[0]) ||
        (PDBLin.Hdr == *(VG_LONG *)PDBAtmRec[1])) {

     if (Num++ < TotalAtm)
       sscanf(PDBLin.Line + 30, "%8f%8f%8f", &Atm -> x, &Atm -> y, &Atm -> z);
     else
       return FALSE;

     Atm = Atm -> Ptr;
    }
  } /* End of while */

  if (Num != TotalAtm)
    return FALSE;

  return TRUE;
}


/**** PDB Saver ****/

VG_BOOL PDBSave(FILE *OUT, ATOMO *AtmIniz, VG_ULONG Tot, VG_WORD Type)
{
  char              AtdlStr[10], *j;
  register ATOMO    *Atm;
  VG_QCHAR          Name;
  VG_UWORD          l;
  VG_WORD           Hdr;

  VG_BOOL           Ret       = TRUE;
  float             Charge    = 0;
  VG_ULONG          k         = 1;
  VG_ULONG          NumConect = 0;
  VG_ULONG          NumRem    = 3;
  VG_UWORD          NumTer    = 0;

  if (fprintf(OUT, "REMARK   4\n" \
                   "REMARK   4 %s\n"\
                   "REMARK   4\n", VEGAHdr) >= 0) {

    switch(Type) {
    case PDB_ATDL:
      for(Atm = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr) {
        AtmToAtdl(AtdlStr, Atm);
        if (fprintf(OUT, "REMARK  78 %5d %8.4f %-4.4s %s",
                    Atm -> Num, Atm -> Charge, Atm -> Pot.C, AtdlStr) > 0) {
          for(l = 0; (Ret) && (l < Atm -> NSost); ++l) {
            if (l) j = " ";
            else j = " (";
            if (fprintf(OUT, j) <= 0) {
              Ret = PrintDosErr();
              break;
            }
            AtmToAtdl(AtdlStr, Atm -> Conn[l]);
            if (fprintf(OUT, AtdlStr) <= 0)
              Ret = PrintDosErr();
          } /* End of for */
          if (Ret) {
            if (l) j = ")\n";
            else j = "\n";
            if (fprintf(OUT, j) < 0)
              Ret = PrintDosErr();
          }
        } else Ret = PrintDosErr();
      } /* End of atom loop */
      NumRem += Tot;
      break;

    case PDB_FAT:
      for(Atm = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr) {
        if (fprintf(OUT, "REMARK  77 EXTRA %5d %-2.2s %-4.4s %8.4f\n",
                    Atm -> Num, Atm -> Elem.C, Atm -> Pot.C, Atm -> Charge) < 0)
          Ret = PrintDosErr();
      } /* End of atom loop */
      NumRem += Tot;
      break;
    } /* End of switch */

    /**** Writes ATOM/HETATM records ****/

    if (Ret) {
      for(Atm = AtmIniz; (Ret) && (Atm); Atm = Atm -> Ptr) {
        if (Type == PDB_NONSTD) Hdr = 0;
        else Hdr = Atm -> Flags & VG_ATMF_HETATM;
        if (fprintf(OUT, "%-6s%5d ", PDBAtmRec[Hdr], k) >= 0) {
          l = Atm -> Name.C[3];
          if (Atm -> Elem.C[1]) {
            if (fprintf(OUT, "%-4.3s", Atm -> Name.C) < 0) {
              Ret = PrintDosErr();
              break;
            }
          } else if (isdigit(l)) {
            Name.L = Atm -> Name.L;
            for(j = Name.C + 3;j > Name.C; --j)
              *j = *(j - 1);
            *Name.C = l;
            if (fprintf(OUT, "%-4.4s", Name.C) < 0) {
              Ret = PrintDosErr();
              break;
            }
          } else if (fprintf(OUT, " %-3.3s", Atm -> Name.C) < 0) {
            Ret = PrintDosErr();
            break;
          }

          if (Type == PDB_PDBQ) Charge = Atm -> Charge;
          if (fprintf(OUT," %-3.3s %c%4.4s    %8.3f%8.3f%8.3f  1.00 %5.2f\n",
                      Atm -> ResName.C, Atm -> ChainID, Atm -> ResSeq.C,
                      Atm -> x, Atm -> y, Atm -> z, Charge) >= 0) {
            if ((Type != PDB_NONSTD) && (Atm -> Flags & VG_ATMF_SEGEND)) {
              ++k;
              if (fprintf(OUT, "TER   %5d      %-3.3s  %4.4s\n",
                          k, Atm -> ResName.C, Atm -> ResSeq.C) >= 0)
                ++NumTer;
              else Ret = PrintDosErr();
            }
            ++k;
          } else Ret = PrintDosErr();
        } else Ret = PrintDosErr();
      }  /* End of atom loop */

      if (Ret) {

        /**** Writes the connectivity ****/

        if (GLOBSW_CONNSAV) {
          for(Atm = AtmIniz;(Ret) && (Atm); Atm = Atm -> Ptr) {
            Hdr = 0;
            for(k = 0;(Ret) && (Atm -> Conn[k]) && (k < MAXBOND); ++k) {
              if (!Hdr) {
                if (fprintf(OUT, "CONECT%5d", Atm -> Num) >= 0) {
                  Hdr = 1;
                  ++NumConect;
                } else Ret = PrintDosErr();
              }
	          if ((Ret) && (fprintf(OUT,"%5d", Atm -> Conn[k] -> Num) < 0))
                Ret = PrintDosErr();
            } /* End of bond order loop */
            if ((Ret) && (Hdr) && (fprintf(OUT,"\n") < 0))
              Ret = PrintDosErr();
          } /* End of atom loop */
        }

        /**** Writes MASTER record ****/

        if ((Ret) && (Type != PDB_NONSTD) &&
            (fprintf(OUT,"MASTER %8d    0    0    0    0    0    0    0%5d%5d%5d    0\n",
                     NumRem, Tot, NumTer, NumConect) < 0))
          Ret = PrintDosErr();
        if ((Ret) && (fprintf(OUT, "END\n") < 0))
          Ret = PrintDosErr();
      }
    }
  } else PrintDosErr();

  return Ret;
}


/**** Seek a trajectory frame ****/

VG_BOOL PDBSeekFrm(TRJINFO *Info, VG_LONG Frames, VG_LONG Mode)
{
  VG_LONG       Offset;

  switch(Mode) {
  case SEEK_CUR:
    Offset = Info -> FrmCur + Frames;
    break;

  case SEEK_END:
    Offset = Info -> Frames - Frames;
    break;

  case SEEK_SET:
    Offset = Frames;
    break;
  } /* End of switch */

  if (Offset < 0)                       Offset = 0;
  if (Offset > (VG_LONG)Info -> Frames) Offset = Info -> Frames - 1;

  if (fseek(Info -> FH, Info -> SeekTab[Offset], SEEK_SET) == 0)
    return TRUE;

  return FALSE;
}

