
/*************************************************
****         VEGA - Interface to zlib         ****
**** Copyright 1996-2001, Alessandro Pedretti ****
*************************************************/


#include <stdio.h>
#include <string.h>

#include "../Zlib/zlib.h"

#ifdef __WIN32__
#include <windows.h>
#endif

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globstr.h"

#define  FALSE                  0
#define  TRUE                   1
#define  GZ_BUFFER_SIZE         65536


/**** Local prototypes ****/

/* Some VEGA definitions are incompatible with the standard windows.h */
/* file and thus can't be used in the following prototypes            */

void         *Alloca(unsigned int);
short        CatErr(unsigned int, ...);
short        GZ_DepackFile(char *, char *);
static short GZ_Error(gzFile *FH);
short        GZ_PackFile(char *, char *);
short        PrintDosErr(void);

#ifdef WIN32
static short GZ_Init(void);
short        GetProcAddresses(HINSTANCE *, LPCSTR, INT, ... );

/**** Library function pointers ****/

int           (WINAPI *VG_gzclose)(gzFile)                        = NULL;
const char *  (WINAPI *VG_gzerror)(gzFile, int *)                 = NULL;
gzFile        (WINAPI *VG_gzopen)(const char *, const char *)     = NULL;
int           (WINAPI *VG_gzread)(gzFile, voidp, unsigned)        = NULL;
int           (WINAPI *VG_gzwrite)(gzFile, const voidp, unsigned) = NULL;

/**** Global variables ****/

extern HINSTANCE                hZlib32;
#else
#  define  VG_gzclose           gzclose
#  define  VG_gzerror           gzerror
#  define  VG_gzopen            gzopen
#  define  VG_gzread            gzread
#  define  VG_gzwrite           gzwrite
#endif


/**** Decompress a file into another ****/

short GZ_DepackFile(char *Src, char *Dest)
{
  char          *Buf;
  FILE          *FO;
  gzFile        FI;
  int           Rd;

  short        Ret = TRUE;

#ifdef WIN32
  if (!GZ_Init())
    return FALSE;
#endif

  if ((Buf = (char *)Alloca(GZ_BUFFER_SIZE))) {
    if ((FI = VG_gzopen(Src, "rb"))) {
      if ((FO = fopen(Dest, "wb"))) {
        while((Ret) && (Rd = VG_gzread(FI, Buf, GZ_BUFFER_SIZE))) {
          if (Rd == -1)
            Ret = GZ_Error(&FI);
          else if (fwrite(Buf, Rd, 1, FO) != 1)
           Ret = PrintDosErr();
        }
        fclose(FO);
      } else Ret = PrintDosErr();
      VG_gzclose(FI);
    } else Ret = FALSE;
  } else Ret = GZ_Error(&FI);

  return Ret;
}


/**** Compress a file into another ****/

short GZ_PackFile(char *Src, char *Dest)
{
  char          *Buf;
  gzFile        FO;
  FILE          *FI;
  int           Rd;

  short        Ret = TRUE;

#ifdef WIN32
  if (!GZ_Init())
    return FALSE;
#endif

  if ((Buf = (char *)Alloca(GZ_BUFFER_SIZE))) {
    if ((FI = fopen(Src, "rb"))) {
      if ((FO = VG_gzopen(Dest, "wb"))) {
        while ((Ret) && (Rd = fread(Buf, 1, GZ_BUFFER_SIZE, FI))) {
          if (VG_gzwrite(FO, Buf, Rd) != Rd)
            Ret = GZ_Error(FO);
        }
        if (ferror(FI)) Ret = PrintDosErr();
        VG_gzclose(FO);
      } else Ret = GZ_Error(FO);
      fclose(FI);
    } else Ret = PrintDosErr();
  } else Ret = FALSE;

  return Ret;
}


/**** Show a gzip error ****/

static short GZ_Error(gzFile *FH)
{
  int           err;

  return CatErr(MSG_ERR_GZIP_ERROR, VG_gzerror(FH, &err));
}

#ifdef WIN32

/**** Library initialization ****/

static short GZ_Init(void)
{
  if (hZlib32)
    return TRUE;

  if (GetProcAddresses(&hZlib32, "zlib32.dll", 5,
                   &VG_gzclose, "gzclose",
                   &VG_gzerror, "gzerror",
                   &VG_gzopen , "gzopen",
                   &VG_gzread , "gzread",
                   &VG_gzwrite, "gzwrite"
                   ) == FALSE)
    return FALSE;

  return TRUE;
}
#endif

