
/********************************************************
****           VEGA - PowerPacker Un/Packer          ****
****        Original code by Marc Espie (1995)       ****
**** Original PowerPacker algorithm by Nico Francois ****
****      Adapted to VEGA by Alessandro Pedretti     ****
********************************************************/


#include <stdio.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"

#define val(p) (((p)[0]<<16) | ((p)[1] << 8) | ((p)[2]))

VG_BOOL PP_DepackFile(char *, char *);
VG_BOOL PP_PackFile(char *, char *);

#ifdef AMIGA

/* AmigaOS uses the high-optimized powerpacker.library without other code */

#include <exec/types.h>
#include <exec/memory.h>
#include <exec/libraries.h>
#include <dos/dos.h>
#include <libraries/ppbase.h>
#include <inline/powerpacker.h>

extern struct Library          *PPBase;


/**** Decompress a file into another ****/

VG_BOOL PP_DepackFile(char *Src, char *Dest)
{
  FILE          *FO;
  UBYTE         *MemBuf;
  VG_ULONG      Err, Size;

  VG_BOOL       Ret = TRUE;

  if (!PPBase) PPBase = (struct Library *)OpenLibrary("powerpacker.library", 0);
  if (PPBase) {
    if (!(Err = ppLoadData(Src, DECR_NONE, MEMF_PUBLIC, &MemBuf, (ULONG *)&Size, NULL))) {
      if ((FO = fopen(Dest, "wb"))) {
        if (fwrite(MemBuf, Size, 1, FO) != 1) Ret = PrintDosErr();
        fclose(FO);
      } else Ret = PrintDosErr();
      FreeMem(MemBuf, Size);
    } else Ret = CatErr(MSG_ERR_PKPP_ERROR, ppErrorMessage(Err));
  } else Ret = CatErr(MSG_ERR_PKPP_NOPPLIB);

  return Ret;
}


/**** Compress a file into another ****/

VG_BOOL PP_PackFile(char *Src, char *Dest)
{
  APTR          BFH, CrunchBuf, CrunchInfo;
  FILE          *FH;
  VG_UBYTE      *MemBuf;
  VG_ULONG      cSize, Err, Size;

  VG_BOOL       Ret = TRUE;

  if (!PPBase) PPBase = (struct Library *)OpenLibrary("powerpacker.library", 0);
  if (PPBase) {
    if (CrunchInfo = ppAllocCrunchInfo(CRUN_GOOD,
                                       SPEEDUP_BUFFMEDIUM, NULL, NULL)) {
      if ((FH = fopen(Src, "rb"))) {
        if ((Size = FileLength(FH))) {
          if ((MemBuf = (VG_UBYTE *)AllocMem(Size, MEMF_PUBLIC))) {
            if (fread(MemBuf, Size, 1, FH) == 1) {
              cSize = ppCrunchBuffer(CrunchInfo, MemBuf, Size);
              if ((cSize != PP_CRUNCHABORTED) && (cSize != PP_BUFFEROVERFLOW)) {
                if ((BFH = (APTR)Open(Dest, MODE_NEWFILE))) {
                  if (ppWriteDataHeader((LONG)BFH, CRUN_GOOD, FALSE, NULL)) {
                  if (Write(BFH, MemBuf, cSize) != cSize) Ret = FALSE;
                  } else Ret = FALSE;
                  Close(BFH);
                } else Ret = FALSE;
              } else Ret = FALSE;
            } else Ret = PrintDosErr();
          } else Ret = CatErr(MSG_ERR_UTILS_OUTOFMEM);
        } else Ret = FALSE;
        fclose(FH);
      } else Ret = PrintDosErr();
      ppFreeCrunchInfo(CrunchInfo);
    } else Ret = FALSE;
  } else Ret = CatErr(MSG_ERR_PKPP_NOPPLIB);

  return Ret;
}

#else
/* Code for non-AmigaOS systems */

/**** Local variables ****/

static VG_ULONG    counter;
static VG_ULONG    shift_in;
static VG_UBYTE    *source;

/**** Local proptotypes ****/

static VG_ULONG get_bits(VG_ULONG);
static void PP_Depack(VG_UBYTE *, VG_UBYTE *, VG_ULONG, VG_ULONG);
static VG_ULONG PP_DepackedLen(VG_UBYTE *, VG_ULONG);


/**** Get a sequence o bits ****/

static VG_ULONG get_bits(VG_ULONG n)
{
  register VG_ULONG        result = 0;
  register VG_ULONG        i;

  for (i = 0; i < n; i++) {
    if (counter == 0) {
      counter = 8;
      shift_in = *--source;
    }
    result = (result<<1) | (shift_in & 1);
    shift_in >>= 1;
    counter--;
  }

  return result;
}


/**** Depack a segment of memory ****/

static void PP_Depack(VG_UBYTE *packed, VG_UBYTE *depacked, VG_ULONG plen,
                      VG_ULONG unplen)
{
  register VG_UBYTE     *dest;
  register int          n_bits;
  register int          idx;
  register VG_ULONG     bytes;
  register int          to_add;
  register VG_ULONG     offset;
  VG_UBYTE              offset_sizes[4];
  register VG_ULONG     i;

  /**** Skip signature ****/

  offset_sizes[0] = packed[4];
  offset_sizes[1] = packed[5];
  offset_sizes[2] = packed[6];
  offset_sizes[3] = packed[7];

  /**** Initialize source of bits ****/

  source = packed + plen - 4;
  dest   = depacked + unplen;

  /**** Skip bits ****/

  get_bits(source[3]);

  /**** Do it forever, i.e., while the whole file isn't unpacked ****/

  while (1) {
    /**** Copy some bytes from the source anyway ****/

    if (get_bits(1) == 0) {
      bytes = 0;
      do {
        to_add = get_bits(2);
        bytes += to_add;
      } while (to_add == 3);
      for (i = 0; i <= bytes; i++)
        *--dest = get_bits(8);
      if (dest <= depacked) return;
    }

    /**** Decode what to copy from the destination file ****/

    idx    = get_bits(2);
    n_bits = offset_sizes[idx];

    /**** Bytes to copy ****/

    bytes = idx + 1;
    if (bytes == 4) {	/* 4 means >=4               */
                        /* and maybe a bigger offset */
      if (get_bits(1) == 0)
        offset = get_bits(7);
      else
        offset = get_bits(n_bits);

      do {
        to_add = get_bits(3);
        bytes += to_add;
      } while (to_add == 7);
    } else offset = get_bits(n_bits);
    for (i = 0; i <= bytes; i++) {
      dest[-1] = dest[offset];
      dest--;
    }
    if (dest <= depacked) return;
  }
}


/**** Depack a PowerPacked file ****/

VG_BOOL PP_DepackFile(char *Src, char *Dest)
{
  FILE          *FI, *FO;
  VG_UBYTE      *PkBuf, *UpBuf;
  VG_ULONG      PkLen, UpLen;

  VG_BOOL       Ret = TRUE;

  if ((FI = fopen(Src, "rb"))) {
    PkLen = FileLength(FI);
    if ((PkBuf = (VG_UBYTE *)Alloca(PkLen))) {

      /**** Read the file ****/

      if (fread(PkBuf, PkLen, 1, FI) == 1) {
        UpLen = PP_DepackedLen(PkBuf, PkLen);
        if (UpLen > PkLen) {
          if ((UpBuf = (VG_UBYTE *)Alloca(UpLen))) {
            if ((FO = fopen(Dest, "wb"))) {

              /**** Unpack the buffered file ****/

              PP_Depack(PkBuf, UpBuf, PkLen, UpLen);

              /**** Write unpacked file ****/

              if (fwrite(UpBuf, UpLen, 1, FO) != 1)
                Ret = PrintDosErr();
              fclose(FO);
            } else Ret = PrintDosErr();
            FREE(UpBuf);
          } else Ret = FALSE;
        } else Ret = CatErr(MSG_ERR_PKPP_TOOSHORT);
      } else Ret = PrintDosErr();
      FREE(PkBuf);
    } else Ret = FALSE;
    fclose(FI);
  } else Ret = FALSE;

  return Ret;
}


/**** Calculate the length of unpacked file ****/

static VG_ULONG PP_DepackedLen(VG_UBYTE *packed, VG_ULONG plen)
{
  counter = 0;

  if (packed[0] != 'P' || packed[1] != 'P' || packed[2] != '2' || packed[3] != '0')
    return 0; /* Not a powerpacker file */

  return val(packed + plen - 4);
}


/**** Compress a file into another ****/

VG_BOOL PP_PackFile(char *Src, char *Dest)
{
  return CatErr(MSG_ERR_PKPP_NOCRUNCH);
}

#endif



