
/*************************************************
****        VEGA - Solvate a molecule         ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/


#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "formats.h"

#define  COVTOL			2.5     /* Overlapping of covalent radius */


/**** Structures ****/

typedef struct {
  float         Gx, Gy, Gz;             /* Coordinates of geometry center */
  float         MaxX, MaxY, MaxZ;       /* Molecule extension             */
} MOLINFO;


/**** Prototypes ****/

void CalcMolInfo(ATOMO *, VG_ULONG, MOLINFO *);


/**** Solvate a molecule ****/

VG_BOOL Solvate(char *ShellFile, ATOMO *InizAtm, VG_ULONG *TotAtm)
{
  ATOMO                 *BegRes, *InizSol, *LastAtom, *PrecAtm, *PrecRes;
  float                 *InizCovRad, *InizCovRadSol;
  float                 *CovRad;
  float                 Mx, My, Mz;
  MOLINFO               MolInfo, SolInfo;
  register ATOMO        *Atm, *Sol;
  VG_ULONG              j, k;

  float                 Nx     = 0.0;
  float                 Ny     = 0.0;
  float                 Nz     = 0.0;
  VG_BOOL               Ret    = TRUE;
  float                 Rad    = Prefs.SOL_RAD;
  VG_ULONG              TotSol = 0;


  if ((InizSol = (ATOMO *)Loader(ShellFile, &TotSol, NULL, 0))) {
    if ((InizCovRad = (float *)AssignCovRad(InizAtm, *TotAtm))) {
      PrintProg(MSG_SOLVATE_SOLVATING);

      /**** Calculate dimensions and geometry center ****/

      CalcMolInfo(InizAtm, *TotAtm, &MolInfo);
      CalcMolInfo(InizSol, TotSol, &SolInfo);

      if ((MolInfo.MaxX <= SolInfo.MaxX) &&
          (MolInfo.MaxY <= SolInfo.MaxY) &&
          (MolInfo.MaxZ <= SolInfo.MaxZ)) {

        /**** Translate the cluster ****/

        Mx = MolInfo.Gx - SolInfo.Gx;
        My = MolInfo.Gy - SolInfo.Gy;
        Mz = MolInfo.Gz - SolInfo.Gz;
        for(Atm = InizSol; Atm; Atm = Atm -> Ptr) {
          Atm -> x += Mx;
          Atm -> y += My;
          Atm -> z += Mz;
        }

        /**** Cut the shell ****/

        switch(Prefs.SOL_SHAPE) {

        case SOL_SHAPE_BOX:
          Rad /= 2.0;
          Mx = MolInfo.Gx + Rad;
          My = MolInfo.Gy + Rad;
          Mz = MolInfo.Gz + Rad;
          Nx = MolInfo.Gx - Rad;
          Ny = MolInfo.Gy - Rad;
          Nz = MolInfo.Gz - Rad;
          break;

        case SOL_SHAPE_SPHERE:
          Rad *= Rad;
          Mx   = MolInfo.Gx;
          My   = MolInfo.Gy;
          Mz   = MolInfo.Gz;
          break;

        case SOL_SHAPE_LAYER:
          Rad *= Rad;
          break;
        }

        BegRes  = InizSol;
        PrecRes = InizSol;
        Sol     = InizSol;
        do {
          if (BegRes -> ResSeq.L != Sol -> ResSeq.L) {
            PrecRes = PrecAtm;
            BegRes  = Sol;
          }
          switch(Prefs.SOL_SHAPE) {
          case SOL_SHAPE_BOX:
            if ((Sol -> x > Mx) || (Sol -> x < Nx) ||
                (Sol -> y > My) || (Sol -> y < Ny) ||
                (Sol -> z > Mz) || (Sol -> z < Nz)) {
              RemoveRes(&BegRes, &PrecRes, &InizSol, &TotSol);
              Sol = BegRes;
            } else {
              PrecAtm = Sol;
              Sol = Sol -> Ptr;
            }
            break;

          case SOL_SHAPE_SPHERE:
            if ((Quad(Sol -> x - Mx) + Quad(Sol -> y - My) + Quad(Sol -> z - Mz)) > Rad) {
              RemoveRes(&BegRes, &PrecRes, &InizSol, &TotSol);
              Sol = BegRes;
            } else {
              PrecAtm = Sol;
              Sol = Sol -> Ptr;
            }
            break;

          case SOL_SHAPE_LAYER:
            k = TRUE;
            for(Atm = InizAtm; (Atm) && (k); Atm = Atm -> Ptr) {
              if (QuadDist(Atm, Sol) <= Rad) {
                PrecRes = BegRes;
                do {
                  BegRes -> Flags |= ATM_FLG_GENERIC;
                  BegRes = BegRes -> Ptr;
                } while((BegRes) && (BegRes -> ResSeq.L == PrecRes -> ResSeq.L));
                Sol = BegRes;
                k   = FALSE;
              }
            }
            if (k) {
              PrecAtm = Sol;
              Sol = Sol -> Ptr;
            }
            break;
          } /* End of switch */
        } while(Sol); /* End of solvent loop */

        /**** Remove layer atoms ****/

        if (Prefs.SOL_SHAPE == SOL_SHAPE_LAYER) {
          PrecAtm = InizSol;
          Sol     = InizSol;
          do {
            if (Sol -> Flags & ATM_FLG_GENERIC) {
              Sol -> Flags &= ATM_FLG_GENERIC - 1;
              PrecAtm = Sol;
              Sol     = Sol -> Ptr;
            } else {
              LastAtom = Sol; /* Sol is a register pointer ! */
              RemoveAtm(&LastAtom, &PrecAtm, &InizSol);
              Sol = LastAtom;
              --TotSol;
            }
          } while(Sol); /* End of solvent loop */
        }

        /**** Removing bumping solvent molecules ****/

        if ((TotSol) && (InizCovRadSol = (float *)AssignCovRad(InizSol, TotSol))) {

          /**** Renumber the atoms of the solvent ****/

          k = 1;
          for(Atm = InizSol; Atm; Atm = Atm -> Ptr)
            Atm -> Num = k++;

          for(Atm = InizAtm; (Atm) && (InizSol); Atm = Atm -> Ptr) {
            CovRad  = InizCovRad;
            BegRes  = InizSol;
            PrecRes = InizSol;
            Sol     = InizSol;
            do {
              if (BegRes -> ResSeq.L != Sol -> ResSeq.L) {
                PrecRes = PrecAtm;
                BegRes  = Sol;
              }
              Mx = (*CovRad + InizCovRadSol[Sol -> Num - 1]) * 1.2;
              if (QuadDist(Atm, Sol) <= Quad(Mx)) {
                RemoveRes(&BegRes, &PrecRes, &InizSol, &TotSol);
                Sol = BegRes;
              } else {
                PrecAtm = Sol;
                Sol = Sol -> Ptr;
              }
            } while(Sol); /* End of solvent loop */
            ++CovRad;
          } /* End of molecule loop */
          FREE(InizCovRadSol);
        } else Ret = FALSE;

        /**** Make the binding between molecule and solvent ****/

        if ((Ret) && (InizSol)) {

          if (GLOBSW_CONNCALC)
             Ret = Connect(InizSol, TotSol);

          if (GLOBSW_OPENGL)
             AssignVdwNew(InizSol, 0, 0);

          for(Atm = InizAtm; Atm; Atm = Atm -> Ptr)
            LastAtom = Atm;
          *TotAtm += TotSol;
          LastAtom -> Ptr = InizSol;

          /**** Renumber the atoms ****/

          j      = 1;
          k      = LastAtom -> Num;
          BegRes = InizSol;
          for(Atm = InizSol; Atm; Atm = Atm -> Ptr) {
            if (BegRes -> ResSeq.L != Atm -> ResSeq.L) {
              BegRes = Atm;
              ++j;
            }
            Atm -> Num = ++k;
            LastAtom = Atm;
          } /* End of solvent loop */
          if (!GLOBSW_STDOUT) CatPrintf(stdout, MSG_SOLVATE_ADDED, j);
          RenRes(InizAtm, 1);
          LastAtm = LastAtom;
        }  else if (!GLOBSW_STDOUT) PrintProg(MSG_ERR_SOLVATE_NOTHING);
      } else Ret = CatErr(MSG_ERR_SOLVATE_SHELLDIM);

/*
      } else Ret = CatErr(MSG_ERR_SOLVATE_READING);
*/

      FREE(InizCovRad);
    } else Ret = FALSE;
    if (!Ret) PDBFree(InizSol);
  } else Ret = CatErr(MSG_ERR_SOLVATE_CANTOPEN);

  return Ret;
}


/**** Calculates geometry center and max dimensions ****/

void CalcMolInfo(register ATOMO *Atm, VG_ULONG TotAtm, register MOLINFO *Info)
{
  float         MinX, MinY, MinZ;

  Info -> Gx   = 0.0;
  Info -> Gy   = 0.0;
  Info -> Gz   = 0.0;
  Info -> MaxX = Atm -> x;
  Info -> MaxY = Atm -> y;
  Info -> MaxZ = Atm -> z;
  MinX         = Atm -> x;
  MinY         = Atm -> y;
  MinZ         = Atm -> z;
  while(Atm) {
    if (Atm -> x > Info -> MaxX) Info -> MaxX = Atm -> x;
    if (Atm -> x < MinX) MinX = Atm -> x;
    if (Atm -> y > Info -> MaxY) Info -> MaxY = Atm -> y;
    if (Atm -> y < MinY) MinY = Atm -> y;
    if (Atm -> z > Info -> MaxZ) Info -> MaxZ = Atm -> z;
    if (Atm -> z < MinZ) MinZ = Atm -> z;
    Info -> Gx += Atm -> x;
    Info -> Gy += Atm -> y;
    Info -> Gz += Atm -> z;
    Atm         = Atm -> Ptr;
  } /* End of atom loop */
  Info -> Gx   /= TotAtm;
  Info -> Gy   /= TotAtm;
  Info -> Gz   /= TotAtm;
  Info -> MaxX  = fabs(Info -> MaxX - MinX) + COVTOL;
  Info -> MaxY  = fabs(Info -> MaxY - MinY) + COVTOL;
  Info -> MaxZ  = fabs(Info -> MaxZ - MinZ) + COVTOL;
}

