
/*************************************************
****    VEGA - Gromacs trajectory analyzer    ****
**** Copyright 1996-2002, Alessandro Pedretti ****
*************************************************/


#ifdef __WIN32__
#  include <windows.h>
#endif

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <malloc.h>
#include "../XdrfLib/xdrf.h"

#ifdef __BORLANDC__
#  pragma hdrstop
#endif

#include "globdef.h"
#include "globvar.h"
#include "globstr.h"
#include "formats.h"


/**** Constants ****/

#define  VG_XTC_HDRSIZE         (13 * 4)
#define  VG_XTC_SEEKOFF         ((13 + 9) * 4)

/**** Local prototypes ****/

static VG_BOOL  XtcCreateTab(TRJINFO *);
static VG_ULONG XtcFrames(XDR *);


#ifdef USE_XDRF_DLL

/**** Library function pointers ****/

static int    (WINAPI *VG_XDRF_3Dfcoord)(XDR *, float *, int *, float *) = NULL;
static int    (WINAPI *VG_XDRF_Close)(XDR *)                             = NULL;
static bool_t (WINAPI *VG_XDRF_Float)(XDR *, float *)                    = NULL;
static bool_t (WINAPI *VG_XDRF_Int)(XDR *, int *)                        = NULL;
static int    (WINAPI *VG_XDRF_Open)(XDR *, const char *, const char *)  = NULL;
static int    (WINAPI *VG_XDRF_Seek)(XDR *, long, int)                   = NULL;

/**** Global variables ****/

extern HINSTANCE               hXdrf;


/**** Initialize the xdrf library ****/

VG_BOOL XdrfInit(void)
{
  if ((hXdrf == NULL) &&
      (GetProcAddresses(&hXdrf, "xdrf.dll", 6,
                   &VG_XDRF_3Dfcoord            , "xdr3dfcoord",
                   &VG_XDRF_Close               , "xdrclose",
                   &VG_XDRF_Float               , "xdr_float",
                   &VG_XDRF_Int                 , "xdr_int",
                   &VG_XDRF_Open                , "xdropen",
                   &VG_XDRF_Seek                , "xdrseek"
                   ) == FALSE))
    return FALSE;
  return TRUE;
}
#else
#  define  VG_XDRF_3Dfcoord       xdr3dfcoord
#  define  VG_XDRF_Close          xdrclose
#  define  VG_XDRF_Float          xdr_float
#  define  VG_XDRF_Int            xdr_int
#  define  VG_XDRF_Open           xdropen
#  define  VG_XDRF_Seek           xdrseek
#endif


/**** Close the trajectory file ****/

void XtcClose(TRJINFO *Info)
{
  if (Info -> FH) {
    VG_XDRF_Close((XDR *)Info -> FH);
    FREE(Info -> FH);
  }
  if (Info -> SeekTab)   FREE(Info -> SeekTab);
  if (Info -> VectorBuf) FREE(Info -> VectorBuf);
}


/**** Crreate the seek table ****/

static VG_BOOL XtcCreateTab(TRJINFO *Info)
{
  int                   k, Skip;
  register VG_ULONG     *Ptr;

  register VG_ULONG     Frames = 0;
  register VG_ULONG     Offset = 0;

  if ((Info -> SeekTab = (VG_ULONG *)Alloca(Info -> Frames * sizeof(VG_ULONG))) != NULL) {
    Ptr = Info -> SeekTab;
    while((VG_XDRF_Seek((XDR *)Info -> FH, VG_XTC_SEEKOFF, SEEK_CUR) == 0) &&
          (VG_XDRF_Int((XDR *)Info -> FH, &Skip)) && (Skip)) {
      if ((k = Skip & 3)) Skip += (4 - k);
      if (VG_XDRF_Seek((XDR *)Info -> FH, Skip, SEEK_CUR) == 0) {
        Offset += VG_XTC_SEEKOFF + Skip + sizeof(int);
        *++Ptr  = Offset;
        ++Frames;
      } else break;
    }

    if (Frames == Info -> Frames) {
      VG_XDRF_Seek((XDR *)Info -> FH, 0, SEEK_SET);
      return TRUE;
    }
  }

  return FALSE;
}


/**** Count the trajectory frames ****/

static VG_ULONG XtcFrames(XDR *FH)
{
  int          k, Skip;

  VG_ULONG     Frames = 0;

  while((VG_XDRF_Seek(FH, VG_XTC_SEEKOFF, SEEK_CUR) == 0) &&
        (VG_XDRF_Int(FH, &Skip)) && (Skip)) {
    if ((k = Skip & 3)) Skip += (4 - k);
    if (VG_XDRF_Seek(FH, Skip, SEEK_CUR) == 0)
      ++Frames;
    else break;
  } /* End of while */

  if (Frames) VG_XDRF_Seek(FH, 0, SEEK_SET);

  return Frames;
}


/**** Load the associated .log file ****/

VG_BOOL XtcLoadLog(FILE *FH, TRJINFO *Info)
{
  char          Buf[256], Str[16];
  float         *Ptr;

  VG_BOOL       Ret = TRUE;
  VG_ULONG      Frm = 0;
  VG_ULONG      k   = 0;

  /**** Count the frames ****/

  while(fgets(Buf, 255, FH)) {
    if (!strncmp(Buf + 11, "Step", 4)) {
      ++Info -> EneFrames;
    }

    /**** Temperature ****/

    if (!strncmp(Buf + 3, "bd_temp", 7))
      sscanf(Buf, "%*s %*s %f", &Info -> Temp);

    /**** Trajectory refresh ****/

    if (!strncmp(Buf + 3, "nstxtcout", 9))
      sscanf(Buf, "%*s %*s %d", &Info -> FrmStep);

    /**** Log refresh ****/

    if (!strncmp(Buf + 3, "nstlog", 6))
      sscanf(Buf, "%*s %*s %d", &Info -> EneStep);

    if (!Info -> EneFrames) k += strlen(Buf);
  } /* End of while */

  if ((Info -> EneFrames) &&
      (Info -> Energy = (float *)Alloca(Info -> EneFrames * sizeof(float)))) {
    Frm = 0;
    Ptr = Info -> Energy;
    fseek(FH, k, SEEK_SET);
    while((Ret) && (fgets(Buf, 255, FH))) {
      if (!strncmp(Buf + 11, "Step", 4)) {
        if (fgets(Buf, 256, FH)) {
          k = 0;
          sscanf(Buf, "%d", &k);

          /**** Start frame and step info ****/

          if (Frm == 1) Info -> StartTime = k;
          else if (Frm == 2) Info -> EneStep = k - Info -> StartTime;

          while(Ret) {
            if (fgets(Buf, 256, FH)) {
              *Str = 0;
              sscanf(Buf, "%15s", Str);
              if ((*Str) && (!strcmp(Str, "LJ"))) {
                if (fgets(Buf, 256, FH)) {
                  sscanf(Buf, "%*s %*s %*s %*s %f", Ptr++);
                  break;
                } else Ret = FALSE;
              }
            } else Ret = FALSE;
          } /* End of while */
        } else Ret = FALSE;
      }
    } /* End of while */
  } else Ret = FALSE;

  if (!Ret) FREE(Info -> Energy);

  return Ret;
}


/**** Open the trajectory file ****/

VG_BOOL XtcOpen(char *FileName, TRJINFO *Info)
{
  char          OldExt[32];
  FILE          *FH;
  float         NumF;
  int           k;
  VG_UWORD      LogFormat;

  VG_BOOL       Ret = FALSE;


#ifdef USE_XDRF_DLL
  if (!XdrfInit()) return FALSE;
#endif

  Info -> SeekTab   = NULL;
  Info -> VectorBuf = NULL;
  if ((Info -> FH = (FILE *)Alloca(sizeof(XDR))) != NULL) {
    if (VG_XDRF_Open((XDR *)Info -> FH, FileName, "r")) {

      /**** Header ****/

      if (VG_XDRF_Int((XDR *)Info -> FH, &k) && (k == 1995)) {

        /**** Number of atoms ****/

        if (VG_XDRF_Int((XDR *)Info -> FH, &k))  {
          Info -> MolAtm    = k;

          /**** Step number & time ****/

          if ((VG_XDRF_Int((XDR *)Info -> FH, &k)) &&
              (VG_XDRF_Float((XDR *)Info -> FH, &NumF))) {
            Info -> StartTime = (VG_ULONG)NumF;
            VG_XDRF_Seek((XDR *)Info -> FH, 0, SEEK_SET);
            if ((Info -> Frames = XtcFrames((XDR *)Info -> FH)) != 0) {
              if (XtcCreateTab(Info)) {
                Ret = TRUE;
              }
            }
          }
        }
      }
    }
  }

  if (Ret) {
    ChangeExt(FileName, "log", OldExt);
    if ((FH = OpenPlot(FileName, &LogFormat)) != NULL) {
      if (LogFormat == FPLOT_GROMACSLOG) XtcLoadLog(FH, Info);
      PkClose(FH);
    }
    ChangeExt(FileName, OldExt, NULL);
  } else XtcClose(Info);

  return Ret;
}


/**** Read one frame ****/

VG_BOOL XtcReadFrm(TRJINFO *Info, register ATOMO *Atm)
{
  float         Precision;
  XYZ           *Ptr;

  if (!Info -> VectorBuf)
     Info -> VectorBuf = (XYZ *)Alloca(sizeof(XYZ) * Info -> MolAtm);

  if ((Info -> VectorBuf) &&
      (VG_XDRF_Seek((XDR *)Info -> FH, VG_XTC_HDRSIZE, SEEK_CUR) == 0)) {
    if (VG_XDRF_3Dfcoord((XDR *)Info -> FH, (float *)(Info -> VectorBuf),
                         (int *)&Info -> MolAtm, &Precision)) {
      Ptr = Info -> VectorBuf;
      while(Atm) {
        Atm -> x = Ptr -> x * 10.0;
        Atm -> y = Ptr -> y * 10.0;
        Atm -> z = Ptr -> z * 10.0;
        Atm = Atm -> Ptr;
        ++Ptr;
      } /* End of while */
      return TRUE;
    }
  }

  return FALSE;
}


/**** Seek a trajectory frame ****/

VG_BOOL XtcSeekFrm(TRJINFO *Info, VG_LONG Frames, VG_LONG Mode)
{
  VG_LONG       Offset;

  switch(Mode) {
  case SEEK_CUR:
    Offset = Info -> FrmCur + Frames;
    break;

  case SEEK_END:
    Offset = Info -> Frames - Frames;
    break;

  case SEEK_SET:
    Offset = Frames;
    break;
  } /* End of switch */

  if (Offset < 0)                       Offset = 0;
  if (Offset > (VG_LONG)Info -> Frames) Offset = Info -> Frames - 1;

  if (VG_XDRF_Seek((XDR *)Info -> FH, Info -> SeekTab[Offset], SEEK_SET) == 0)
    return TRUE;

  return FALSE;
}


